/* TOS 4.04 web server using FreeRTOS / LWIP
 * Didier Mequignon 2008-2011, e-mail: aniplay@wanadoo.fr
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* ------------------------ System includes ------------------------------- */
#include <string.h>
#include <math.h>
#include "m68k_disasm.h"
#define Colour ColourFvdi
#include "fb.h"
#undef Colour

/* ------------------------ FreeRTOS includes ----------------------------- */
#include "../freertos/FreeRTOS.h"
#include "../freertos/task.h"
#include "../freertos/queue.h"

/* ------------------------ Project includes ------------------------------ */
#include "config.h"
#include "../../include/version.h"
#include "../../include/fire.h"
#undef __MBAR
#include "net.h"
#ifdef MCF5445X
#include "flash.h"
#else
#ifdef MCF547X
#include "amd_flash.h"
#else /* MCF548X */
#include "intel_c3_flash.h"
#endif /* MCF547X */
#endif /* MCF5445X */
#include "get.h"
#include "gif.h"

/* ------------------------ lwIP includes --------------------------------- */
#include "sockets.h"

/* ------------------------ BDOS includes ----------------------------------*/
#undef min
/* cannot use TOS trap out of CF68KLIB !!! */
#include "../bdos/bdos.h"
/* hope than the TOS update this variables ! */
#define Tgetdate() *(unsigned short *)0x1176
#define Tgettime() *(unsigned short *)0x1178

#ifdef NETWORK
#ifdef LWIP

#ifdef MCF5445X /* MCF5445X has no FPU */
#define WEB_LIGHT
#endif

/* ------------------------ Defines --------------------------------------- */
/* Connection socket */
#define TIMEOUT                 2 /* sec */
#define READBUF_SIZE            6400
#define WRITEBUF_SIZE           32000
#define SOCK_BUF_SIZE           32000
#define TASKS_SERVER            5

#define BUF_SIZE 1024
#define MAX_LENGTH 1000

/* Picture, curves */
#define PICTURE_WIDTH 640
#define PICTURE_HEIGHT 480
#define NB_COLONNES 20
#define NB_LIGNES 16
#define NB_COURBES 4
#define NB_IMAGES 5

/* Trigger, curves */
#define CODE_DECLENCHE    -1
#define CODE_FIN_DECLENCH -2
#define CODE_STOP         -3

/* The port on which we listen. */
#define webHTTP_PORT            80

/* the http versions we support */
enum http_version {VERSION_UNKNOWN, VERSION_10, VERSION_11};

#define SIZE_TOS_FLASH      0x00100000 // last 65KB for TOS parameters

typedef struct sock_msg_struct
{
  int sock;
  int reserved[3];
} sock_msg;

typedef struct sock_conn_struct
{
  int sock;
  int errno;
  int version;
  int content_length;
  char boundary[256];
  char *response;
  int response_size;
  int response_max_size;
  char *buf_write;
  int offset_buf_write;
} sock_conn;

typedef struct http_list_struct
{
  const char *method;
  const char *url;
  void (*handle_url)(sock_conn *conn, char *request);
  const char *content_type;
} http_list;

typedef struct http_status_struct
{
  int code;
  const char *text;
} http_status;

extern short colours[];
extern short video_found;
extern int errno;
extern void conout_debug(int c);
extern void conws_debug(char *buf);
extern void ltoa(char *buf, long n, unsigned long base);
extern void ftoa(float x, int f, char *buf);
extern long atol(const char *text);
extern float atof(const char *text);

static char *types1[] = { "HEXA", "BIT", "BYTE", "SHORT", "LONG", "FLOAT", "DOUBLE", NULL };
static char *types2[] = { "BYTE", "SHORT", "LONG", "FLOAT", "DOUBLE", NULL };

#ifndef WEB_LIGHT
static char *types3[] = { "BYTE", "SHORT", "LONG", "FLOAT", "DOUBLE", "TASKS", NULL }; /* same type2 + TASKS */

static unsigned char work_gif[] = {
71,73,70,56,57,97,100,0,136,0,230,126,0,0,0,0,16,8,5,20,16,2,20,19,19,36,20,0,
33,24,16,51,24,0,39,39,1,36,33,21,55,35,1,51,36,22,53,53,0,55,55,22,36,36,36,52,
41,33,55,50,34,53,53,53,66,24,0,88,24,0,70,37,2,72,53,4,68,49,18,86,41,0,85,50,
1,82,55,28,66,45,57,66,55,39,70,57,50,108,46,0,99,53,2,69,69,0,66,66,24,90,74,
8,88,70,24,87,87,0,71,69,39,68,66,56,74,82,41,83,67,52,84,84,41,90,90,53,113,69,
4,113,91,10,110,91,44,126,119,4,118,117,40,69,69,69,83,74,70,88,85,71,87,87,87,
99,85,70,116,101,77,124,120,71,123,113,93,107,107,107,115,107,107,121,116,104,
121,119,117,131,49,0,134,69,3,144,74,0,149,83,0,157,105,32,173,84,0,189,90,0,170,
101,0,189,107,0,146,117,86,132,120,109,137,126,112,163,126,84,206,99,0,198,115,
0,140,140,0,152,148,0,149,149,31,167,167,0,181,173,0,188,188,0,154,150,72,152,
144,92,140,140,107,132,132,126,156,132,107,174,147,94,173,140,107,165,140,115,
188,150,111,173,173,83,186,186,90,189,189,107,217,128,0,233,140,0,254,152,0,255,
156,16,255,173,41,201,159,116,206,165,123,214,169,123,244,183,118,206,206,0,218,
218,0,235,235,0,254,254,0,201,201,100,140,139,138,148,140,133,156,148,140,151,
151,151,165,156,156,165,165,156,184,167,151,169,169,169,181,181,173,185,185,185,
218,173,132,222,181,132,198,189,189,234,186,140,226,185,144,253,202,151,202,202,
202,214,206,206,217,217,217,239,222,198,235,235,235,0,0,0,0,0,0,33,255,11,78,69,
84,83,67,65,80,69,50,46,48,3,1,0,0,0,33,249,4,9,10,0,126,0,44,1,0,1,0,96,0,126,
0,0,8,254,0,253,8,28,72,176,160,193,131,2,243,212,168,177,132,73,13,56,8,35,74,
156,72,177,162,69,137,112,150,56,57,195,145,99,153,22,16,47,138,28,73,114,100,
156,141,29,83,158,41,131,163,164,203,151,48,253,208,56,67,102,163,153,36,77,206,
220,204,217,50,166,207,159,8,225,148,33,211,36,9,77,39,70,207,36,97,82,198,76,
20,160,80,161,182,56,195,228,230,25,39,78,148,112,116,66,198,104,153,144,81,195,
150,20,202,164,73,147,50,71,149,152,209,169,228,236,25,26,98,227,142,196,65,180,
140,218,180,29,181,26,101,178,71,174,95,138,45,186,150,201,121,52,233,153,179,
70,201,16,249,203,248,32,28,166,75,215,234,76,130,146,227,82,157,45,26,107,30,
8,135,133,8,207,73,112,58,145,220,17,237,74,22,155,55,119,86,66,70,242,80,37,90,
85,150,97,130,58,53,99,57,75,72,171,220,109,70,9,109,57,182,253,182,104,170,210,
76,111,22,200,89,24,231,104,198,67,18,41,193,227,206,100,210,154,35,19,17,8,32,
228,144,146,6,142,148,6,15,238,254,82,133,30,61,106,20,50,226,155,84,176,33,7,
162,156,56,113,250,230,129,243,226,67,146,181,75,202,67,205,211,214,227,130,24,
105,248,193,70,26,108,232,177,199,30,112,164,209,70,95,57,184,16,194,70,249,233,
231,19,13,70,153,193,194,3,125,185,32,69,30,4,113,232,135,28,3,218,0,71,30,48,
120,192,2,112,18,194,196,134,8,42,52,0,86,12,126,184,33,16,68,112,244,21,160,30,
46,200,232,71,30,58,166,8,83,123,5,217,144,3,88,123,216,16,128,136,2,1,233,227,
95,112,184,224,2,4,49,52,0,192,148,3,176,1,71,14,40,46,249,215,129,113,76,233,
101,12,72,106,217,88,31,49,120,57,37,27,98,222,102,228,148,5,12,49,68,0,0,184,
112,131,136,125,165,25,149,13,102,106,96,7,30,120,204,0,128,12,5,120,233,66,150,
118,194,132,167,151,122,242,137,135,29,10,204,113,7,2,83,54,16,3,121,133,186,116,
232,148,14,236,169,40,30,87,240,57,5,0,147,230,209,71,165,46,237,1,167,151,85,
108,90,3,24,138,50,250,37,169,254,37,93,10,192,3,155,14,97,130,28,157,242,41,131,
153,104,194,122,145,169,160,74,153,234,162,68,228,96,2,31,82,104,58,135,0,3,184,
0,0,4,190,94,148,71,3,108,12,0,192,0,123,218,1,157,28,199,238,49,44,30,15,12,0,
1,28,54,68,107,209,129,82,2,160,1,159,85,236,193,7,31,199,242,145,131,162,126,
2,208,128,30,230,90,148,135,181,234,242,153,195,187,240,190,11,199,28,236,122,
9,109,190,20,193,65,229,0,116,208,225,6,192,27,0,60,131,174,102,122,136,112,68,
121,0,32,133,181,97,12,1,192,187,126,152,240,49,31,53,44,10,169,151,96,93,140,
208,11,105,76,41,3,14,83,190,43,242,199,105,224,225,177,153,48,170,140,49,9,83,
190,32,69,204,240,198,44,197,12,51,63,96,66,12,57,232,28,17,28,8,184,57,3,24,82,
236,33,71,14,219,237,209,135,9,97,204,224,166,2,85,204,80,167,210,5,17,17,6,29,
96,204,49,4,30,29,179,202,46,214,54,147,61,135,24,25,128,109,80,30,87,132,49,133,
214,55,220,113,246,254,166,124,14,113,135,13,67,204,96,68,24,85,200,93,80,30,106,
219,33,131,27,105,236,205,247,16,221,193,176,40,30,96,24,78,80,31,50,240,25,6,
129,112,64,160,246,162,97,152,64,2,28,108,72,17,134,174,150,19,196,198,6,97,212,
48,32,27,114,64,96,130,9,40,52,0,129,20,113,188,94,67,232,148,166,238,71,29,46,
216,192,70,27,185,123,199,157,149,164,231,222,134,13,113,251,126,184,136,109,128,
8,7,233,32,74,145,160,244,229,58,95,80,145,196,179,17,199,123,210,119,151,187,
28,203,167,172,61,150,105,200,209,158,149,221,69,63,164,250,233,155,239,124,12,
200,195,238,253,236,38,192,33,71,29,245,247,168,189,31,147,130,29,249,224,96,130,
56,232,239,86,108,168,67,245,228,231,59,54,216,64,10,176,115,67,130,240,39,189,
220,73,193,6,254,211,158,28,88,230,61,239,129,200,128,112,248,30,233,114,144,189,
255,13,196,129,82,144,67,119,220,96,165,58,80,143,13,108,112,1,3,157,151,7,48,
117,199,126,176,107,3,236,114,160,157,81,25,100,84,254,9,156,30,8,105,132,48,57,
184,160,1,46,208,33,236,72,199,134,28,12,96,7,43,104,195,65,246,32,193,21,204,
192,138,43,80,131,135,102,88,41,57,196,96,0,226,122,210,17,31,48,3,46,116,129,
11,53,232,144,20,124,176,5,51,118,225,141,103,68,194,12,252,128,32,149,201,161,
8,23,216,193,5,58,0,6,48,208,33,8,111,20,2,135,242,48,3,36,192,241,144,136,132,
206,215,242,165,6,55,158,209,8,139,226,65,15,206,104,2,31,8,225,144,102,116,36,
28,185,208,59,132,249,0,142,65,16,194,8,238,96,135,10,60,192,140,147,68,164,16,
122,96,70,36,8,193,140,91,232,1,190,236,104,200,55,6,1,4,0,232,84,21,170,176,133,
46,164,242,144,59,216,65,11,86,240,4,44,172,96,11,43,160,128,210,136,128,201,10,
116,140,79,159,244,229,38,121,144,0,4,100,1,13,104,192,66,22,90,208,131,44,148,
240,98,55,64,228,22,198,192,39,58,68,147,149,66,184,0,1,122,80,130,21,208,0,155,
239,68,1,22,208,208,147,52,89,105,144,82,254,220,17,28,106,176,130,80,34,33,5,
23,104,99,23,200,9,6,33,116,160,7,172,4,100,23,132,80,130,39,96,243,154,104,104,
1,54,147,38,166,58,126,8,118,123,80,195,12,132,208,203,46,108,33,1,19,8,101,10,
14,218,133,47,128,129,7,66,216,65,2,224,232,3,26,64,20,13,79,160,129,75,183,25,
32,49,161,40,13,123,168,195,12,124,160,201,51,94,114,147,177,108,227,14,108,249,
198,110,182,0,5,8,64,0,3,92,138,133,121,174,192,7,189,18,19,68,106,80,75,68,90,
21,147,93,8,66,71,223,216,2,166,174,224,14,116,152,65,1,28,138,6,21,156,113,5,
179,244,81,128,106,96,213,45,108,97,7,35,29,169,91,175,26,74,91,94,19,11,223,154,
1,13,90,160,133,178,194,81,8,94,243,209,136,126,186,80,30,80,0,2,16,104,195,244,
66,248,164,11,4,161,167,61,216,130,43,105,128,215,114,234,53,11,239,124,194,47,
223,232,131,37,201,97,171,92,144,129,197,230,134,3,5,164,96,171,11,125,236,1,134,
69,7,13,188,243,154,79,184,254,64,79,187,80,79,253,124,22,142,91,168,237,210,54,
144,2,55,10,1,9,33,16,67,223,30,144,5,204,182,0,1,178,181,42,18,118,224,35,57,
32,65,146,92,8,194,34,39,2,135,6,164,160,7,65,240,1,29,240,112,7,5,244,160,3,2,
32,64,48,103,91,212,42,165,72,10,239,68,3,13,66,48,90,138,192,231,13,123,155,2,
6,4,122,85,171,238,96,74,8,168,105,112,242,208,130,121,102,161,169,115,36,8,28,
220,208,151,197,42,22,34,125,57,155,29,134,144,2,36,112,1,9,146,173,175,45,185,
32,0,51,53,32,58,112,40,38,54,181,240,4,18,28,196,128,2,153,174,26,222,128,135,
26,32,212,183,168,181,106,10,18,144,46,131,109,38,125,126,88,67,95,177,57,209,
130,40,201,15,82,168,65,26,106,144,227,58,169,16,7,132,245,101,85,17,201,5,3,60,
43,7,55,40,211,148,8,37,22,21,66,36,15,81,141,194,12,210,64,229,52,120,72,14,97,
176,3,143,29,199,39,59,4,74,127,16,97,3,97,93,89,223,45,152,201,76,54,112,65,254,
90,197,82,167,60,212,193,15,33,249,218,30,160,12,156,52,236,9,12,97,128,100,171,
192,117,45,130,216,160,170,29,224,66,79,139,124,230,66,199,160,189,4,145,195,12,
232,151,135,233,78,100,15,109,176,178,31,106,43,135,40,208,128,9,88,105,1,12,26,
128,128,116,5,0,109,4,59,219,3,26,208,73,12,184,177,7,67,62,99,10,10,205,234,0,
228,172,32,82,224,1,15,44,96,128,20,76,64,6,92,20,112,28,6,82,178,150,236,97,56,
164,177,208,169,236,53,128,204,213,205,102,119,120,117,162,83,201,133,11,176,178,
11,255,172,48,171,167,237,2,24,198,153,7,64,232,193,5,126,192,1,32,28,225,7,186,
53,72,141,6,98,103,202,177,161,5,148,33,195,85,88,32,237,35,65,96,74,103,219,155,
155,118,125,144,28,60,118,161,0,16,192,176,11,29,128,6,240,240,204,3,216,144,30,
228,112,3,1,29,225,8,59,176,64,10,14,126,240,30,32,90,220,110,202,86,118,14,176,
128,5,0,220,6,178,98,213,222,2,208,201,32,37,32,5,210,46,244,0,254,72,93,160,52,
180,120,74,1,144,195,129,234,112,160,28,176,1,8,64,176,0,183,143,32,4,29,236,0,
8,81,61,136,28,112,112,58,60,136,97,8,39,179,48,98,115,240,69,47,13,96,3,120,160,
131,223,134,144,235,216,177,90,92,97,42,8,28,248,5,170,3,153,220,5,121,112,185,
16,142,192,129,30,44,156,7,28,144,0,16,242,121,144,60,224,96,83,14,0,64,191,161,
68,181,6,61,233,204,30,88,116,21,100,64,130,13,64,135,139,39,143,83,14,242,176,
230,41,30,138,13,123,136,195,12,116,124,165,56,236,224,8,60,216,1,7,184,46,129,
9,244,32,231,5,145,55,159,210,62,237,51,47,160,42,249,177,129,1,247,0,195,136,
200,74,78,13,160,119,69,54,102,160,24,216,65,91,113,176,94,4,104,222,109,132,115,
64,241,101,31,64,0,166,224,168,189,237,170,242,1,16,128,8,76,83,6,17,140,8,167,
144,151,186,153,142,24,128,25,14,168,202,4,73,67,233,183,171,184,28,164,96,245,
71,184,128,183,179,125,132,23,24,100,15,252,50,129,205,182,254,127,5,47,5,224,
0,224,247,0,108,212,189,146,50,44,197,9,248,202,169,28,30,46,144,119,79,9,2,82,
34,59,65,44,202,25,52,169,220,6,123,184,1,9,182,75,7,18,136,87,7,136,119,115,63,
240,3,23,240,112,74,166,0,54,99,7,83,176,39,50,144,59,177,193,28,88,17,26,101,
209,17,45,49,34,24,81,104,214,39,18,112,80,112,68,247,61,14,160,55,28,192,1,4,
48,1,7,183,120,61,192,3,48,16,17,122,144,117,53,48,7,19,51,3,85,176,1,2,49,21,
58,129,105,100,64,126,84,65,126,112,65,17,238,103,116,236,135,17,73,2,127,105,
0,1,59,32,107,39,120,112,40,181,3,6,240,102,19,161,6,182,51,0,48,80,66,48,160,
27,91,17,26,166,113,6,74,144,48,133,166,95,35,1,121,123,112,120,222,246,3,12,215,
3,28,144,129,231,194,35,146,38,7,56,184,27,41,193,4,20,33,43,125,246,18,56,69,
16,59,32,4,217,230,109,71,144,2,22,96,1,249,99,17,211,131,34,117,66,131,108,168,
18,92,120,125,251,6,0,254,193,7,19,10,176,135,59,160,3,72,64,134,137,199,3,5,119,
17,122,112,61,2,49,133,131,168,18,29,151,104,103,230,2,80,209,23,156,231,2,2,176,
112,9,183,3,63,192,3,76,54,18,109,80,25,153,120,6,105,20,17,74,230,37,63,72,18,
92,152,7,22,144,135,139,119,4,24,80,18,9,52,16,44,176,134,131,88,27,8,225,44,130,
18,22,115,40,16,50,160,120,38,88,136,21,97,62,46,192,21,173,168,20,73,98,2,6,240,
2,91,68,116,178,7,0,204,8,20,53,149,3,4,144,0,64,176,139,49,193,6,110,16,5,77,
64,133,42,65,6,42,247,0,33,152,120,83,163,114,7,98,68,185,6,19,96,193,6,4,64,45,
227,152,3,125,161,6,85,145,137,102,208,0,21,32,107,215,117,4,29,64,69,136,133,
83,122,240,77,97,1,22,179,120,46,26,2,65,105,118,133,108,40,2,22,48,1,28,240,3,
135,215,3,164,99,2,115,160,1,49,0,105,142,214,69,238,119,65,169,55,20,153,200,
2,8,213,3,51,119,4,56,144,7,83,96,51,54,80,254,7,248,23,45,114,96,38,1,144,6,34,
226,2,196,49,136,77,240,3,212,199,3,98,216,1,198,34,6,38,32,53,10,105,39,123,16,
139,1,112,68,82,224,2,5,160,4,101,208,20,232,184,18,55,7,115,8,197,117,216,182,
3,54,144,100,190,194,52,145,2,120,116,68,63,48,48,26,58,49,149,102,192,3,7,151,
109,61,128,149,61,0,4,36,0,120,139,69,2,13,34,60,54,208,29,125,167,26,61,120,97,
6,33,7,44,0,141,153,168,2,99,184,150,107,121,112,13,8,1,84,247,116,146,178,141,
114,209,50,84,194,69,31,96,6,55,72,19,108,200,2,12,7,4,25,217,112,235,49,109,135,
136,114,16,80,109,127,193,6,70,23,146,3,33,3,85,33,145,42,225,4,131,201,117,12,
183,135,106,48,114,54,64,63,138,117,79,80,230,114,121,7,0,250,40,23,233,2,1,162,
57,16,121,32,2,234,166,110,102,32,145,78,240,1,121,224,3,72,136,150,4,121,148,
18,241,30,223,225,125,77,150,46,202,121,16,35,224,27,196,177,22,179,177,0,48,112,
32,121,254,64,134,19,112,139,136,215,144,17,145,3,94,162,151,220,120,42,7,83,17,
105,208,151,191,153,4,11,224,0,244,67,101,112,96,4,100,200,1,146,212,3,226,232,
18,10,51,37,20,229,19,140,89,117,20,81,36,46,16,7,21,224,1,7,0,117,138,69,66,158,
232,7,85,112,112,167,104,145,41,248,18,192,242,44,63,145,159,160,66,17,108,16,
139,57,181,129,10,164,62,9,36,7,229,146,52,84,80,130,29,32,131,36,49,103,121,80,
67,40,151,136,36,1,157,20,17,139,83,82,7,46,231,114,27,170,62,42,148,37,67,240,
93,19,80,18,56,112,152,246,2,158,21,145,3,112,50,0,19,33,7,121,215,0,110,192,33,
252,163,62,27,122,161,54,218,1,6,32,127,23,33,158,244,131,81,64,129,125,83,146,
107,121,48,108,3,176,119,73,18,61,24,151,6,88,82,58,236,49,16,70,48,1,80,42,18,
95,52,41,187,73,18,135,34,164,17,225,116,215,162,108,4,17,3,164,163,88,49,160,
29,235,151,36,63,1,101,36,224,36,211,3,19,25,163,159,111,202,47,254,231,121,16,
108,208,153,226,178,29,185,147,104,76,232,19,7,162,63,110,80,151,6,164,7,146,246,
163,94,146,138,116,100,3,252,34,167,215,103,101,221,33,7,220,65,111,121,240,154,
9,250,23,121,0,65,21,113,42,1,48,69,252,210,0,107,218,33,121,144,62,223,20,60,
79,210,24,109,0,32,21,225,37,251,25,98,177,72,158,20,113,162,126,112,3,42,180,
7,221,225,7,46,32,46,2,129,170,113,17,157,8,225,37,57,199,6,45,198,172,8,1,28,
224,51,60,160,72,81,23,148,102,113,97,129,22,225,37,41,228,69,196,40,168,35,65,
58,243,49,53,189,162,95,49,224,2,49,224,97,113,1,34,152,106,16,225,42,114,141,
138,17,183,57,16,36,244,61,83,163,167,59,50,45,28,103,170,209,195,6,121,240,174,
253,217,137,119,169,115,72,243,145,3,81,163,234,147,61,192,49,143,126,17,171,233,
243,102,22,211,131,84,226,163,51,114,176,202,70,167,119,106,125,42,90,30,39,58,
90,60,4,1,47,224,176,5,225,64,81,82,175,126,96,3,137,53,103,59,98,32,66,5,145,
174,183,147,7,113,0,99,51,146,6,81,227,178,54,214,44,29,82,39,175,138,179,8,65,
101,209,18,16,0,0,33,249,4,9,10,0,126,0,44,1,0,3,0,96,0,126,0,0,8,254,0,253,8,
28,72,176,160,193,131,3,149,48,73,146,68,73,25,132,16,35,74,156,72,177,162,68,
134,102,204,156,57,99,166,201,2,37,22,67,138,28,73,146,201,198,147,40,155,48,33,
201,178,165,75,129,28,203,112,116,184,49,9,153,50,77,252,192,121,201,179,39,66,
39,102,22,158,113,226,36,201,70,37,74,204,224,164,225,179,169,79,163,76,204,24,
37,227,68,201,198,38,100,172,58,97,177,199,169,87,150,39,136,42,161,58,52,171,
70,156,70,207,36,193,241,181,173,69,56,44,202,80,141,122,134,170,213,154,106,207,
48,89,225,182,175,68,56,42,138,6,221,72,52,173,222,50,83,23,248,93,124,16,142,
67,37,77,100,158,41,3,249,100,80,163,102,60,48,222,60,240,4,11,164,34,24,146,225,
168,241,36,208,186,32,57,111,102,194,34,180,228,162,73,154,160,52,45,66,53,99,
22,73,156,204,62,89,166,244,73,38,76,106,219,238,43,34,233,238,186,78,154,16,77,
41,98,229,240,182,34,204,16,45,109,70,201,130,3,46,92,192,136,17,227,128,136,209,
147,147,254,60,111,139,24,188,25,22,14,92,176,21,152,199,207,158,28,49,2,124,175,
57,190,105,26,164,39,89,12,136,225,39,141,148,28,54,164,97,131,13,108,228,176,
71,29,1,12,192,66,70,39,212,215,83,20,44,112,212,196,7,108,248,177,95,14,3,237,
97,67,0,49,72,193,134,28,57,192,193,198,8,177,201,225,224,75,56,176,192,196,2,
54,116,229,135,11,110,64,32,99,12,13,0,96,227,0,108,184,48,192,78,253,241,119,
226,75,0,22,20,67,28,117,236,145,135,20,54,38,25,131,13,49,240,248,163,91,123,
184,144,96,3,3,36,105,99,147,46,72,241,164,95,114,180,184,7,28,86,2,176,36,28,
112,184,184,165,87,54,36,217,71,12,97,178,81,64,146,46,152,120,166,79,105,118,
25,128,141,5,12,49,196,157,234,217,80,102,3,49,164,49,39,79,105,166,153,164,6,
118,224,129,199,12,0,200,240,166,141,113,14,250,210,30,1,24,106,35,162,138,226,
97,135,2,116,220,129,128,141,128,106,41,41,75,3,90,233,64,162,153,226,113,133,
162,83,136,41,69,30,254,125,84,56,170,72,70,86,153,100,21,169,214,0,70,166,155,
42,9,193,172,34,229,97,41,0,15,164,58,132,9,114,172,170,168,12,86,178,161,7,176,
22,81,42,102,141,184,106,74,4,4,38,248,33,5,170,115,4,208,0,9,0,64,208,7,180,21,
229,209,0,27,85,14,144,168,29,82,188,103,2,31,123,84,139,199,3,3,64,0,135,13,237,
145,59,209,30,123,212,8,128,6,138,86,177,7,31,124,188,203,71,14,153,50,10,64,3,
7,234,75,81,30,182,2,140,71,14,4,239,97,66,28,124,192,49,71,192,73,218,235,240,
68,96,2,48,192,0,116,208,17,3,31,93,154,192,6,31,126,212,176,172,149,249,126,12,
81,30,0,72,81,101,24,67,16,204,7,0,6,3,224,178,29,159,38,233,164,204,8,189,144,
134,141,50,224,64,176,141,239,218,152,6,30,67,132,25,131,153,68,23,68,68,30,224,
2,240,130,20,4,87,1,70,211,38,72,49,3,179,196,154,224,99,213,6,229,81,5,28,13,
232,57,3,24,237,198,145,67,14,237,246,97,66,24,51,232,169,64,254,21,51,160,125,
80,30,68,132,65,7,24,115,12,129,7,206,187,106,90,197,221,80,15,62,135,24,25,248,
157,246,21,97,76,145,247,13,119,24,158,170,162,67,220,97,195,16,51,24,17,70,21,
146,23,148,71,226,118,200,224,70,26,154,111,62,68,26,112,192,160,41,30,96,148,
78,80,31,50,40,26,70,26,108,192,1,65,226,154,134,97,2,9,34,74,17,134,162,48,216,
78,16,27,27,132,81,3,27,188,203,129,173,9,40,52,0,129,20,112,180,193,123,13,194,
139,170,188,64,117,184,64,96,27,113,244,46,133,20,188,147,201,70,249,109,216,16,
249,247,166,251,249,97,239,34,202,193,6,246,105,216,127,47,252,6,105,216,134,253,
113,144,67,0,123,7,187,242,201,193,13,54,112,3,255,8,146,3,57,228,79,14,34,130,
3,236,254,23,34,57,56,16,130,11,28,72,12,232,55,191,56,152,224,131,112,144,67,
29,56,168,192,12,250,161,67,31,58,32,28,46,22,66,100,177,161,14,246,195,94,68,
134,38,51,54,216,192,67,7,100,3,27,62,232,66,56,148,79,10,254,9,140,8,213,100,
38,7,163,173,239,67,229,139,3,153,2,40,34,0,61,203,132,126,176,161,20,28,168,62,
56,212,161,126,58,116,1,13,5,50,68,131,108,81,82,121,88,18,236,230,247,161,54,
124,40,7,16,200,193,184,8,210,158,113,237,0,9,61,224,65,28,83,144,2,63,228,225,
75,250,146,131,11,26,224,2,29,126,72,68,5,26,192,14,86,208,6,59,26,146,139,91,
224,194,14,228,40,71,36,112,161,11,2,121,34,185,246,192,134,6,52,224,5,3,138,1,
26,177,181,133,46,4,33,8,46,240,129,16,186,192,5,56,246,32,8,93,72,101,42,225,
24,4,247,124,113,84,93,153,64,10,8,48,1,2,24,96,8,70,0,3,18,82,25,132,20,144,50,
8,61,216,165,42,135,153,202,45,112,81,102,117,216,65,34,73,185,5,42,104,138,7,
163,228,130,0,78,217,73,98,90,19,146,85,195,65,15,84,233,72,20,224,97,14,19,56,
192,35,119,96,205,32,8,65,152,195,108,101,213,108,128,202,84,246,64,5,1,216,216,
16,170,208,201,109,18,83,8,65,254,80,65,15,90,208,130,21,112,65,5,117,28,84,239,
218,147,135,66,218,17,14,53,88,65,7,134,185,5,1,128,142,115,143,180,231,42,103,
73,3,52,160,65,11,104,104,65,15,176,224,130,57,225,209,15,246,147,195,30,212,48,
3,33,84,83,162,171,180,3,29,52,229,3,119,146,114,7,9,152,192,3,74,208,2,139,182,
64,11,43,120,2,26,214,179,37,19,57,144,13,121,152,1,6,170,201,133,30,244,32,5,
143,252,100,42,189,48,6,218,33,33,8,112,228,66,10,144,176,133,45,140,64,167,22,
181,169,69,71,176,134,57,209,173,101,23,224,193,22,224,200,131,32,60,146,11,9,
16,0,29,39,208,129,71,142,1,12,61,216,1,5,46,160,74,33,172,0,11,89,125,194,10,
180,240,132,44,60,1,169,43,144,164,131,160,119,194,30,112,1,152,232,76,165,34,
135,89,84,195,118,193,151,143,77,165,16,176,240,132,22,156,224,4,40,200,2,26,104,
128,215,16,136,181,11,66,152,65,23,135,83,160,79,6,161,154,215,188,102,39,205,
58,204,21,208,224,9,88,24,254,65,21,140,224,128,7,224,53,163,158,116,172,15,126,
132,61,101,94,19,159,166,125,164,53,15,43,204,201,102,33,11,80,216,24,30,170,112,
130,21,88,84,5,170,60,101,23,106,112,162,15,73,116,172,61,32,0,1,150,52,32,27,
32,192,0,142,101,236,54,159,218,215,214,49,119,179,104,192,2,57,85,89,84,21,156,
72,15,114,64,237,22,72,32,40,132,72,1,2,29,104,167,98,131,217,5,6,180,206,4,37,
72,239,113,63,128,90,85,226,232,68,54,64,45,23,136,0,50,27,84,64,191,135,221,129,
230,174,160,1,157,98,1,11,39,168,64,129,21,155,2,0,32,160,190,195,201,195,8,182,
112,78,79,38,143,32,112,112,67,87,200,148,61,56,0,241,193,192,236,65,226,102,208,
129,11,60,128,1,19,72,192,105,175,201,3,2,36,169,1,207,129,67,115,159,80,217,14,
60,224,32,74,60,230,64,108,72,5,31,108,76,12,50,48,43,23,170,154,218,232,134,233,
87,155,201,159,31,138,128,85,1,247,141,32,89,98,143,20,106,144,134,173,213,96,
39,68,80,148,254,24,42,32,132,82,150,178,202,93,216,165,143,195,36,39,183,80,209,
142,31,58,1,86,179,208,2,8,168,33,95,105,0,131,29,106,240,130,121,86,33,12,148,
179,67,1,28,176,210,26,164,0,170,238,20,110,106,123,208,129,48,133,235,5,49,251,
138,139,236,231,135,25,100,53,171,248,250,80,127,194,64,106,196,13,65,12,67,0,
195,16,28,0,128,0,172,1,14,43,24,101,116,19,75,204,4,88,218,74,49,200,116,65,234,
60,146,61,220,192,6,113,56,161,123,242,0,133,39,120,70,4,159,105,1,12,218,54,5,
5,240,76,208,171,34,220,157,240,133,161,28,160,150,181,170,189,117,152,56,228,
149,16,118,101,15,51,8,3,127,246,208,2,50,148,38,43,34,184,147,162,236,32,134,
227,109,238,1,11,35,136,6,80,123,1,46,72,90,149,23,208,182,165,59,122,16,43,204,
96,7,28,0,66,69,202,4,82,41,144,32,115,96,104,67,11,114,211,4,134,48,228,0,76,
147,129,0,70,134,7,58,36,202,112,240,230,55,23,33,219,5,223,222,83,223,97,26,128,
247,254,8,146,2,29,36,192,0,29,192,192,16,216,51,145,66,15,193,14,47,119,0,4,188,
147,4,201,16,70,0,97,50,129,170,142,103,184,23,240,90,32,109,104,231,22,166,122,
86,36,216,26,228,82,51,93,10,126,208,3,14,52,253,7,71,216,193,33,17,34,7,13,184,
155,114,203,5,128,7,152,112,19,142,56,193,3,183,158,1,30,140,112,241,42,200,202,
32,26,16,102,15,4,112,129,11,204,249,214,245,242,87,146,2,208,71,232,1,145,13,
64,56,2,15,44,176,3,168,31,225,8,41,56,123,218,112,128,7,84,167,90,81,38,128,128,
7,144,45,128,235,232,27,87,154,43,192,217,12,210,0,181,6,33,223,250,118,193,6,
209,40,247,86,167,129,95,252,210,3,129,142,0,132,14,0,252,239,70,5,130,65,15,50,
132,194,201,75,120,72,15,19,2,52,101,184,48,104,220,32,195,186,53,4,128,77,53,
13,89,9,2,252,170,3,160,222,99,131,188,119,224,168,71,248,1,7,36,0,4,193,179,103,
0,1,208,19,170,240,160,129,216,91,233,0,4,56,213,170,77,254,240,130,169,163,24,
238,55,144,66,9,151,156,134,242,11,74,122,54,50,80,29,72,48,32,54,64,128,0,61,
208,123,211,143,208,1,9,68,160,249,5,217,131,173,100,176,82,101,177,218,250,2,
160,2,101,224,1,108,19,3,237,35,65,163,5,1,150,198,93,249,71,67,34,82,7,10,200,
47,54,48,7,97,224,6,114,48,0,4,96,1,164,199,1,80,199,1,58,192,1,29,208,69,46,96,
35,100,7,53,136,23,123,7,192,117,28,145,4,54,16,62,58,244,74,33,99,105,54,128,
16,121,224,64,255,35,16,112,240,107,252,34,118,135,131,3,4,192,1,4,240,119,22,
144,119,64,160,3,66,176,1,7,161,128,0,48,3,48,71,7,184,82,5,183,38,0,34,80,28,
230,182,17,76,160,4,73,112,98,118,36,88,4,161,132,97,50,131,141,81,7,249,146,7,
189,163,45,218,179,7,49,32,3,137,114,5,3,144,0,58,192,3,81,7,77,63,0,4,59,176,
69,250,103,35,10,96,130,134,99,7,27,192,36,113,209,27,190,225,112,73,160,17,102,
112,0,253,241,74,254,58,1,119,33,177,7,2,146,35,249,147,6,185,83,5,29,96,0,9,160,
129,64,192,1,71,96,84,72,136,16,208,227,2,96,48,58,48,103,7,38,112,71,17,194,17,
87,216,16,212,1,30,134,40,17,185,103,35,63,71,17,21,146,6,240,97,3,46,176,1,59,
144,2,64,80,114,71,128,4,41,16,117,17,0,134,17,225,2,38,128,0,137,7,1,34,213,105,
199,33,29,13,113,18,74,48,114,253,115,39,95,200,18,60,194,105,114,0,135,59,64,
122,127,151,124,28,48,1,55,240,48,123,16,7,102,228,2,130,194,6,54,119,18,100,0,
30,204,232,140,187,102,105,1,224,18,159,119,76,215,152,124,65,240,119,60,208,1,
9,64,0,206,7,17,58,52,69,249,178,7,165,120,28,40,97,6,147,103,16,108,18,141,61,
33,40,22,64,71,59,112,141,63,192,119,60,176,3,221,184,136,188,35,16,40,0,144,187,
49,0,193,120,101,78,33,139,108,98,0,215,184,3,22,224,116,60,176,122,20,225,64,
203,83,142,22,169,0,2,33,7,38,96,0,152,134,131,157,7,254,0,227,199,19,32,86,126,
152,8,146,127,23,130,35,145,7,79,148,3,11,98,145,249,33,82,15,192,1,28,240,144,
32,162,71,191,231,20,31,85,38,47,160,124,189,248,3,17,25,18,33,36,16,5,114,0,78,
128,146,187,193,2,13,80,1,41,96,143,247,23,130,110,176,71,112,226,21,46,66,38,
88,99,0,19,64,127,163,85,17,81,98,3,47,32,2,186,1,148,30,97,1,19,0,130,166,183,
3,33,194,147,14,52,124,109,113,118,49,160,93,192,56,18,232,23,0,240,209,2,167,
1,144,102,192,118,29,192,1,41,16,127,22,144,7,194,130,47,70,194,48,110,113,118,
90,70,18,114,160,36,2,162,73,35,224,27,199,225,1,17,96,122,70,21,117,114,112,3,
14,48,4,5,192,6,148,180,150,79,130,134,33,151,35,46,86,6,88,153,31,11,201,3,202,
247,119,56,176,7,105,54,4,48,112,36,172,121,34,112,16,52,97,2,32,114,16,3,189,
97,145,74,0,117,214,152,119,43,192,6,76,40,3,48,212,71,112,160,30,226,99,67,176,
195,133,156,225,59,218,254,214,0,112,16,3,16,16,29,232,184,27,101,208,119,121,
23,71,71,192,152,63,224,3,88,18,147,33,7,40,32,198,24,71,179,109,27,20,69,128,
18,22,163,161,20,199,81,6,112,152,137,63,0,135,28,176,72,63,176,1,214,103,37,1,
0,1,125,228,23,108,96,105,3,240,138,108,80,115,225,185,27,42,144,141,163,25,143,
71,112,3,233,50,160,86,98,32,110,209,121,28,162,107,92,196,0,55,49,26,179,201,
2,217,216,159,153,248,119,63,48,1,57,144,32,24,74,160,95,33,7,49,137,136,126,128,
2,64,113,159,199,209,4,126,7,120,217,216,1,41,32,3,85,98,73,220,17,164,109,32,
5,222,89,144,183,6,100,246,1,141,66,243,23,44,160,27,15,186,17,102,32,2,56,144,
141,125,39,132,26,0,82,20,241,30,57,144,53,25,234,19,241,217,49,18,161,33,46,176,
0,68,113,156,224,81,6,44,240,1,59,97,4,77,55,1,26,168,119,30,250,48,92,26,46,61,
17,131,160,18,17,108,96,164,46,192,2,231,200,17,100,192,2,30,208,0,54,192,157,
254,36,48,5,77,103,122,76,167,147,139,8,153,48,154,36,129,217,18,157,199,157,16,
97,164,54,130,3,34,160,2,34,224,1,12,176,123,208,99,3,3,32,40,70,128,122,60,48,
1,59,32,3,34,129,3,182,162,145,64,162,164,72,122,16,141,218,38,221,181,62,3,68,
38,62,215,46,62,240,119,68,169,0,117,32,18,57,32,38,58,244,2,142,58,41,169,138,
101,127,163,164,86,162,127,154,23,7,1,4,67,22,68,32,46,226,3,63,208,149,187,42,
18,9,10,125,182,18,0,239,73,42,115,183,150,232,199,142,162,18,82,22,100,65,247,
146,3,249,50,4,59,16,1,3,89,17,35,8,40,52,50,0,196,35,163,19,65,51,73,146,173,
3,113,129,150,6,1,105,208,30,108,240,63,30,178,175,246,3,32,192,56,4,19,112,123,
33,97,46,70,99,34,252,2,65,247,178,62,112,160,7,249,90,17,189,122,35,211,74,16,
26,146,170,160,50,177,4,97,128,245,67,164,36,160,161,121,32,3,222,233,18,121,144,
6,49,32,62,55,192,47,105,227,33,20,161,164,233,122,254,135,86,50,0,24,91,16,158,
58,50,3,160,35,112,128,33,47,66,139,196,250,18,100,242,31,36,251,33,187,154,167,
244,122,16,86,66,146,174,25,114,216,73,177,30,114,3,22,4,1,220,177,19,16,192,71,
221,167,178,95,33,44,68,26,167,6,49,119,203,19,147,7,70,17,86,148,63,77,210,34,
57,75,35,0,75,176,62,1,7,194,146,174,68,107,35,198,88,7,37,171,160,193,150,165,
105,208,62,117,0,32,185,230,7,214,211,0,110,171,121,59,219,20,216,211,64,20,97,
125,198,56,67,26,202,69,115,35,64,75,18,108,190,3,153,76,194,29,125,25,73,243,
83,131,8,17,123,102,187,107,154,52,53,84,41,32,1,20,36,194,182,19,68,106,3,125,
219,20,58,84,16,14,251,64,135,228,133,236,136,179,51,164,121,142,43,16,180,24,
66,221,25,69,114,50,178,35,243,168,94,81,126,34,68,186,144,41,16,55,176,157,51,
169,143,36,16,3,236,231,34,208,3,67,225,115,80,135,148,7,191,219,23,144,169,61,
118,104,177,78,99,17,78,11,31,47,32,65,34,72,21,3,114,64,164,12,195,35,88,171,
26,95,98,65,245,170,132,3,128,47,33,209,6,49,128,3,13,128,179,122,16,37,227,107,
69,88,10,69,2,193,29,187,39,65,245,229,34,35,147,180,11,148,6,245,98,38,206,226,
30,249,8,191,32,133,191,253,81,184,179,18,16,0,0,59,
};
#endif

static unsigned char red_led_gif[] = {
71,73,70,56,57,97,10,0,10,0,230,0,0,248,132,129,171,51,23,187,46,22,240,36,31,
247,115,112,245,81,80,245,85,84,241,38,34,204,39,21,215,34,19,226,31,19,246,104,
102,200,73,43,213,34,19,246,107,102,244,68,68,247,116,112,248,138,135,221,65,40,
248,143,136,197,72,42,228,187,149,243,51,50,229,232,219,236,26,18,191,44,22,228,
29,19,197,160,125,215,222,209,182,47,23,205,206,186,248,182,159,167,54,24,197,
41,21,209,165,130,233,241,238,239,232,193,246,109,106,231,27,18,247,123,119,238,
30,24,210,35,20,226,30,19,199,40,21,193,43,21,246,181,161,243,49,48,244,72,72,
178,49,23,243,57,57,238,200,169,210,203,175,217,34,20,214,68,40,239,240,228,244,
63,63,238,209,173,235,240,224,221,206,174,238,245,234,247,250,250,191,172,140,
249,160,149,177,50,23,232,27,18,195,42,21,195,159,121,238,29,22,250,166,152,241,
40,37,200,205,186,243,56,55,248,134,131,248,149,142,248,176,157,241,193,167,190,
157,121,227,233,222,241,41,39,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,33,249,4,0,0,0,0,0,44,0,0,0,0,10,0,10,0,0,7,98,128,79,79,60,
54,50,31,56,57,130,79,59,75,45,62,72,4,5,58,79,35,74,68,73,0,16,6,49,7,23,36,17,
0,39,37,5,71,7,24,34,19,4,11,6,15,46,3,64,9,18,14,47,55,22,69,40,42,41,33,53,21,
78,3,67,26,52,8,25,29,66,77,38,10,13,8,65,2,63,1,30,79,51,43,44,2,48,1,32,61,138,
28,27,12,20,76,70,130,129,0,59,
};

static unsigned char green_led_gif[] = {
71,73,70,56,57,97,10,0,10,0,230,78,0,0,143,13,0,148,14,0,134,13,0,190,18,0,224,
21,37,255,58,0,155,14,2,226,23,19,243,40,0,173,16,165,244,186,38,225,59,0,169,
16,0,132,12,85,252,106,0,200,19,0,153,14,0,193,18,0,179,17,20,244,41,111,182,127,
0,215,20,0,180,17,0,159,15,98,180,114,208,237,229,14,238,35,206,237,226,0,176,
16,154,202,171,38,160,54,63,255,84,0,202,19,0,161,15,0,170,16,98,176,113,75,255,
96,124,246,144,183,225,202,0,211,20,0,162,15,0,138,13,0,168,16,0,207,19,0,219,
21,0,139,13,38,150,52,0,191,18,130,223,150,0,147,14,38,167,54,12,236,33,38,152,
53,0,186,17,62,244,83,27,251,48,101,252,122,52,255,73,182,235,202,167,220,186,
191,240,211,0,187,18,98,195,115,147,244,168,40,255,61,150,201,168,0,206,19,44,
255,65,134,204,152,98,181,114,0,150,14,0,163,15,112,217,131,86,255,107,0,141,13,
70,255,91,59,255,80,170,213,188,223,235,243,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,33,249,4,1,0,0,78,0,44,0,0,0,0,10,0,10,0,64,7,98,128,
78,78,25,10,56,14,37,58,130,78,68,16,1,0,45,2,13,20,78,54,8,51,7,21,15,53,9,40,
50,27,75,31,57,5,19,7,43,3,59,72,17,61,18,34,71,6,49,74,24,78,60,73,36,76,64,8,
4,48,78,63,67,5,55,26,4,66,3,28,62,11,44,39,32,47,22,12,23,70,30,38,9,42,33,6,
142,41,2,65,130,77,69,52,46,35,29,130,129,0,59,
};

static char menu_html[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB - Menu</title>\n"
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE - Menu</title>\n"
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE - Menu</title>\n"
#endif
#endif
    "<script language='javascript'>\n"
    "<!-- \n"
    "var pathname = location.pathname; \n"
    "var myDomain = pathname.substring(0,pathname.lastIndexOf('/')) +'/'; \n"
    "var date_exp = new Date(); \n"
    /* cookie MCF5485 for 30 days */
    "date_exp.setTime(date_exp.getTime()+(30*24*3600*1000)); \n"
    "var navigateur = navigator.appName; \n"
    "var plateforme = navigator.platform; \n"
    "var BulleBill = false; \n"
    "var Bold = \"<b>\"; \n"
    "var NoBold = \"</b>\"; \n"
    "if(navigateur == \"Microsoft Internet Explorer\") \n"
    "  BulleBill = true; \n"
    "if(plateforme != \"Win32\") \n"
    "  BulleBill = false; \n"
    "if(BulleBill) Bold = NoBold = \"\"; \n"
    "var bulleStyle = null \n"
    "if(!document.layers && !document.all && !document.getElementById) \n"
    "  event=\"chut\"; \n"
    "function bulle(msg,evt,hauteur) { \n"
    "  var xfenetre,yfenetre,xpage,ypage,element=null; \n"
    "  var offset=15; \n"
    "  var bulleWidth=150; \n"
    "  if(BulleBill) return; \n"
    "  if(!hauteur) hauteur=40; \n"
    "  if(document.layers) { \n"
    "    bulleStyle=document.layers['tip']; \n"
    "    bulleStyle.document.write('<layer bgColor=\"#ffffdd\" ' \n"
    "    +'style=\"width:150px;border:1px solid black;color:black\">' \n"
    "    + msg + '</layer>' ); \n"
    "    bulleStyle.document.close(); \n"
    "    xpage = evt.pageX; ypage = evt.pageY; \n"
    "    xfenetre = xpage; yfenetre = ypage; \n"    
    "  } else if(document.all) { \n"
    "    element=document.all['tip'] \n"
    "    xfenetre = evt.x; yfenetre = evt.y; \n"    
    "    xpage=xfenetre; ypage=yfenetre; \n"
    "    if(document.body.scrollLeft) xpage = xfenetre + document.body.scrollLeft; \n"
    "    if(document.body.scrollTop) ypage = yfenetre + document.body.scrollTop; \n"
    "  } else if (document.getElementById) { \n"
    "    element=document.getElementById('tip') \n"
    "    xfenetre = evt.clientX; yfenetre = evt.clientY; \n"
    "    xpage=xfenetre; ypage=yfenetre; \n"  
    "    if(evt.pageX) xpage = evt.pageX; \n"
    "    if(evt.pageY) ypage = evt.pageY ; } \n"
    "  if(element) { \n"
    "    bulleStyle=element.style; \n"
    "    element.innerHTML=msg; } \n"
    "  if(bulleStyle) { \n"
    "    if (xfenetre > bulleWidth+offset) xpage=xpage-bulleWidth-offset; \n"
    "    else xpage=xpage+15; \n"
    "    if( yfenetre > hauteur+offset ) ypage=ypage-hauteur-offset; \n"
    "    bulleStyle.width=bulleWidth; \n"
    "    if(typeof(bulleStyle.left)=='string') { \n"
    "        bulleStyle.left=xpage+'px'; bulleStyle.top=ypage+'px'; \n" 
    "    } else { \n"
    "      bulleStyle.left=xpage; bulleStyle.top=ypage; } \n"
    "    bulleStyle.visibility=\"visible\"; } } \n"
    "function fin_bulle() { \n"
    "  if(bulleStyle) bulleStyle.visibility=\"hidden\"; } \n"
    "function getCookieVal(offset) { \n"
    "  var endstr = document.cookie.indexOf (\";\", offset); \n"
    "  if(endstr==-1) \n"
    "    endstr=document.cookie.length; \n"
    "  return unescape(document.cookie.substring(offset, endstr)); } \n"
    "function GetCookie(name) { \n"
    "  var arg = name+\"=\"; \n"
    "  var alen = arg.length; \n"
    "  var clen = document.cookie.length; \n"
    "  var i=0; \n"
    "  while(i<clen) { \n"
    "    var j=i+alen; \n"
    "    if(document.cookie.substring(i, j)==arg) \n"
    "      return getCookieVal(j); \n"
    "    i = document.cookie.indexOf(\" \",i)+1; \n"
    "    if(i==0) break; } \n"
    "  return null; } \n"
    "function SetCookie(name, value) { \n"
    "  var argv = SetCookie.arguments; \n"
    "  var argc = SetCookie.arguments.length; \n"
    "  var expires = (argc > 2) ? argv[2] : null; \n"
    "  var path = (argc > 3) ? argv[3] : null; \n"
    "  var domain = (argc > 4) ? argv[4] : null; \n"
    "  var secure = (argc > 5) ? argv[5] : false; \n"
    "  document.cookie = name+\"=\"+escape(value)+ \n"
    "   ((expires==null) ? \"\" : (\"; expires=\"+expires.toGMTString()))+ \n"
    "   ((path==null) ? \"\" : (\"; path=\"+path))+ \n"
    "   ((domain==null) ? \"\" : (\"; domain=\"+domain))+ \n"
    "   ((secure==true) ? \"; secure\" : \"\"); } \n"
    "function cherche(name) { \n"
    "  var cookie = GetCookie(\"MCF5485\"); \n"
    "  if(cookie==null) \n"
    "    return null; \n"
    "  recherche = cookie.indexOf(name); \n"
    "  if(recherche==-1) \n"
    "    return null; \n"
    "  var extrait = cookie.slice(recherche); \n"
    "  var variable = extrait.slice(extrait.indexOf('_')+1, extrait.indexOf('|')); \n"
    "  return variable; } \n"
    "function sauve() { \n"
#ifndef WEB_LIGHT
    "  var SignTrig=0,i,SelC,SelC2,SelC3,SelC4; \n"
    "  for(i=0;i<document.forms[0].SignTrig.length;i++) { \n"
    "    if(document.forms[0].SignTrig[i].checked) \n"
    "      SignTrig = i; } \n"
    "  if(document.forms[0].SelC.checked) SelC = 1; else SelC = 0; \n"
    "  if(document.forms[0].SelC2.checked) SelC2 = 1; else SelC2 = 0; \n"
    "  if(document.forms[0].SelC3.checked) SelC3 = 1; else SelC3 = 0; \n"
    "  if(document.forms[0].SelC4.checked) SelC4 = 1; else SelC4 = 0; \n"
#endif
    "  SetCookie(\"MCF5485\", \n"
    "   document.forms[0].LectMem.name + \"_\" + document.forms[0].LectMem.selectedIndex + \"|\" \n"
    "   + document.forms[0].OffsetLect.name + \"_\" + document.forms[0].OffsetLect.value + \"|\" \n"
    "   + document.forms[0].LengthLect.name + \"_\" + document.forms[0].LengthLect.value + \"|\" \n"
    "   + document.forms[0].Rafraichir.name + \"_\" + document.forms[0].Rafraichir.value + \"|\" \n"
    "   + document.forms[0].ModifMem.name + \"_\" + document.forms[0].ModifMem.selectedIndex + \"|\" \n"
    "   + document.forms[0].IndexModif.name + \"_\" + document.forms[0].IndexModif.value + \"|\" \n"
#ifndef WEB_LIGHT
    "   + document.forms[0].CMem.name + \"_\" + document.forms[0].CMem.selectedIndex + \"|\" \n"
    "   + document.forms[0].IndexLect.name + \"_\" + document.forms[0].IndexLect.value + \"|\" \n"
    "   + document.forms[0].SelC.name + \"_\" + SelC + \"|\" \n"
    "   + document.forms[0].ValeurMini.name + \"_\" + document.forms[0].ValeurMini.value + \"|\" \n"
    "   + document.forms[0].ValeurMaxi.name + \"_\" + document.forms[0].ValeurMaxi.value + \"|\" \n"
    "   + document.forms[0].TempsC.name + \"_\" + document.forms[0].TempsC.selectedIndex + \"|\" \n"
    "   + document.forms[0].CMem2.name + \"_\" + document.forms[0].CMem2.selectedIndex + \"|\" \n"
    "   + document.forms[0].IndexLect2.name + \"_\" + document.forms[0].IndexLect2.value + \"|\" \n"
    "   + document.forms[0].SelC2.name + \"_\" + SelC2 + \"|\" \n"
    "   + document.forms[0].CMem3.name + \"_\" + document.forms[0].CMem3.selectedIndex + \"|\" \n"
    "   + document.forms[0].IndexLect3.name + \"_\" + document.forms[0].IndexLect3.value + \"|\" \n"
    "   + document.forms[0].SelC3.name + \"_\" + SelC3 + \"|\" \n"
    "   + document.forms[0].CMem4.name + \"_\" + document.forms[0].CMem4.selectedIndex + \"|\" \n"
    "   + document.forms[0].IndexLect4.name + \"_\" + document.forms[0].IndexLect4.value + \"|\" \n"
    "   + document.forms[0].SelC4.name + \"_\" + SelC4 + \"|\" \n"
    "   + document.forms[0].TrigMem.name + \"_\" + document.forms[0].TrigMem.selectedIndex + \"|\" \n"
    "   + document.forms[0].IndexTrig.name + \"_\" + document.forms[0].IndexTrig.value + \"|\" \n"
    "   + document.forms[0].SignTrig[0].name + \"_\" + SignTrig + \"|\" \n"
    "   + document.forms[0].ValeurTrig.name + \"_\" + document.forms[0].ValeurTrig.value + \"|\" \n"
    "   + document.forms[0].PreTrig.name + \"_\" + document.forms[0].PreTrig.value + \"|\" \n"
#endif
#ifdef DBUG
    "   + document.forms[0].OffsetDis.name + \"_\" + document.forms[0].OffsetDis.value + \"|\" \n"
    "   + document.forms[0].LengthDis.name + \"_\" + document.forms[0].LengthDis.value + \"|\" \n"
#endif
    "   , date_exp); } \n"
    "function remplissage() { \n"
    "  var LectMem = cherche(\"LectMem\"); \n"
    "  if(LectMem != null) \n"
    "    document.forms[0].LectMem.selectedIndex = parseInt(LectMem); \n"
    "  var OffsetLect = cherche(\"OffsetLect\"); \n"
    "  if(OffsetLect != null) \n"
    "    document.forms[0].OffsetLect.value = OffsetLect; \n"
    "  var LengthLect = cherche(\"LengthLect\"); \n"
    "  if(LengthLect != null) \n"
    "    document.forms[0].LengthLect.value = LengthLect; \n"    
    "  var Rafraichir = cherche(\"Rafraichir\"); \n"
    "  if(Rafraichir != null) \n"
    "    document.forms[0].Rafraichir.value = Rafraichir; \n"
    "  var ModifMem = cherche(\"ModifMem\"); \n"
    "  if(ModifMem != null) \n"
    "    document.forms[0].ModifMem.selectedIndex = parseInt(ModifMem); \n"
    "  var IndexModif = cherche(\"IndexModif\"); \n"
    "  if(IndexModif != null) \n"
    "    document.forms[0].IndexModif.value = IndexModif; \n"
#ifndef WEB_LIGHT
    "  var CMem = cherche(\"CMem\"); \n"
    "  if(CMem != null) \n"
    "    document.forms[0].CMem.selectedIndex = parseInt(CMem); \n"
    "  var IndexLect = cherche(\"IndexLect\"); \n"
    "  if(IndexLect != null) \n"
    "    document.forms[0].IndexLect.value = IndexLect; \n"
    "  var SelC = cherche(\"SelC\"); \n"
    "  if(SelC != null) { \n"
    "    if(SelC==1) document.forms[0].SelC.checked = true; \n"
    "    else document.forms[0].SelC.checked = false; } \n"
    "  var ValeurMini = cherche(\"ValeurMini\"); \n"
    "  if(ValeurMini != null) \n"
    "    document.forms[0].ValeurMini.value = ValeurMini; \n"
    "  var ValeurMaxi = cherche(\"ValeurMaxi\"); \n"
    "  if(ValeurMaxi != null) \n"
    "    document.forms[0].ValeurMaxi.value = ValeurMaxi; \n"
    "  var TempsC = cherche(\"TempsC\"); \n"
    "  if(TempsC != null) \n"
    "    document.forms[0].TempsC.selectedIndex = parseInt(TempsC); \n"
    "  var CMem2 = cherche(\"CMem2\"); \n"
    "  if(CMem2 != null) \n"
    "    document.forms[0].CMem2.selectedIndex = parseInt(CMem2); \n"
    "  var IndexLect2 = cherche(\"IndexLect2\"); \n"
    "  if(IndexLect2 != null) \n"
    "    document.forms[0].IndexLect2.value = IndexLect2; \n"
    "  var SelC2 = cherche(\"SelC2\"); \n"
    "  if(SelC2 != null) { \n"
    "    if(SelC2==1) document.forms[0].SelC2.checked = true; \n"
    "    else document.forms[0].SelC2.checked = false; } \n"
    "  var CMem3 = cherche(\"CMem3\"); \n"
    "  if(CMem3 != null) \n"
    "    document.forms[0].CMem3.selectedIndex = parseInt(CMem3); \n"
    "  var IndexLect3 = cherche(\"IndexLect3\"); \n"
    "  if(IndexLect3 != null) \n"
    "    document.forms[0].IndexLect3.value = IndexLect3; \n"
    "  var SelC3 = cherche(\"SelC3\"); \n"
    "  if(SelC3 != null) { \n"
    "    if(SelC3==1) document.forms[0].SelC3.checked = true; \n"
    "    else document.forms[0].SelC3.checked = false; } \n"
    "  var CMem4 = cherche(\"CMem4\"); \n"
    "  if(CMem4 != null) \n"
    "    document.forms[0].CMem4.selectedIndex = parseInt(CMem4); \n"
    "  var IndexLect4 = cherche(\"IndexLect4\"); \n"
    "  if(IndexLect4 != null) \n"
    "    document.forms[0].IndexLect4.value = IndexLect4; \n"
    "  var SelC4 = cherche(\"SelC4\"); \n"
    "  if(SelC4 != null) { \n"
    "    if(SelC4==1) document.forms[0].SelC4.checked = true; \n"
    "    else document.forms[0].SelC4.checked = false; } \n"
    "  var TrigMem = cherche(\"TrigMem\"); \n"
    "  if(TrigMem != null) \n"
    "    document.forms[0].TrigMem.selectedIndex = parseInt(TrigMem); \n"
    "  var IndexTrig = cherche(\"IndexTrig\"); \n"
    "  if(IndexTrig != null) \n"
    "    document.forms[0].IndexTrig.value = IndexTrig; \n"
    "  var SignTrig = cherche(\"SignTrig\"); \n"
    "  if(SignTrig != null) \n"
    "    document.forms[0].SignTrig[parseInt(SignTrig)].checked = true; \n"
    "  var ValeurTrig = cherche(\"ValeurTrig\"); \n"
    "  if(ValeurTrig != null) \n"
    "    document.forms[0].ValeurTrig.value = ValeurTrig; \n"
    "  var PreTrig = cherche(\"PreTrig\"); \n"
    "  if(PreTrig != null) \n"
    "    document.forms[0].PreTrig.value = PreTrig; \n"
#endif
#ifdef DBUG
    "  var OffsetDis = cherche(\"OffsetDis\"); \n"
    "  if(OffsetDis != null) \n"
    "    document.forms[0].OffsetDis.value = OffsetDis; \n"
    "  var LengthDis = cherche(\"LengthDis\"); \n"
    "  if(LengthDis != null) \n"
    "    document.forms[0].LengthDis.value = LengthDis; \n"
#endif
    "} \n"
    "// --> \n"
    "</script>\n"    
    "</head>\n";
static char menu[] =
    "<body bgcolor=cyan text=black onLoad='remplissage()'>\n"
    "<div id='tip' style='position:absolute; visibility:hidden; z-index:800; "
    "background:#ffa; width:150px; height:auto; padding:4px; text-align:left; "
    "border:2px solid black;'>\n"
    "Il a quelques navigateurs o&ugrave; les bulles ne fonctionnent pas...</div>\n"  
    "<p><center><font color=blue size=5><i><b>"
#ifdef MCF5445X
    "M54455EVB"
#else
#ifdef MCF547X
    "FIREBEE"
#else /* MCF548X */
    "M5484LITE"
#endif
#endif    
    "</b></i></font><br>\n<i>Boot version ";
static char menu0[] =
    "</i><br><hr size=1 color=blue><br></center>\n"
    "<p><h3>Menu :</h3>\n"
    "<form action='/mem.html' method='get'>\n";
static char menu1[] =
    "<p><script language='javascript'>\n"
    "<!-- \n"
    "  var message1 = \"Allows to see memory from a type of size with a refresh rate of 1 to 99 seconds. \"; \n"
    "  if(BulleBill) document.write(\"<span title='\" + message1 + \"'>\"); \n"
    "  else document.write(\"<span>\"); \n"
    "// --> \n"
    "</script>\n"
    "<b onMouseOver='this.style.color=\"blue\"; bulle(message1, event);' onMouseOut='this.style.color=\"black\"; fin_bulle();'> \n"
    "Reading memory : </b></span>\n"
    "<select name='LectMem'>\n";
static char menu2[] =
    "</select>\n Start address $<input type='text' value='0' name='OffsetLect' size=8 maxlength=8>\n"
    " Length <input type='int' value='16' name='LengthLect' size=3 maxlength=3>\n"
    " Refresh rate <input type='int' value='1' name='Rafraichir' size=2 maxlength=2> S\n"
    "<input type='submit' value='OK' name='ChoixLect' onClick='sauve()'>\n"
    "<p><script language='javascript'>\n"
    "<!-- \n"
    "  var message2 = \"Allows to change memory from his address. After his modification, the value from the reading page is displayed in bold. \"; \n"
    "  if(BulleBill) document.write(\"<span title='\" + message2 + \"'>\"); \n"
    "  else document.write(\"<span>\"); \n"
    "// --> \n"
    "</script>\n"
    "<b onMouseOver='this.style.color=\"blue\"; bulle(message2, event);' onMouseOut='this.style.color=\"black\"; fin_bulle();'> \n"
    "Change memory : </b></span>\n"
    "<select name='ModifMem'>\n";
static char menu3[] =
    "</select>\n Address $<input type='text' name='IndexModif' size=8 maxlength=8>\n"
    " Value <input type='text' name='ValeurModif' size=8 maxlength=10>\n"
    "<input type='submit' value='OK' name='ChoixModif' onClick='sauve()'>\n"
#ifndef WEB_LIGHT
    "<p><table border=0 align=right>\n"
    "<tr><td bgcolor=indianred>\n"
    "<select name='CMem'>\n";
static char menu4[] =
    "</select>\n Address $<input type='text' name='IndexLect' size=8 maxlength=8>\n"
    " Red curve &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; <input type='checkbox' value='OK' name='SelC' checked></td>\n"
    "<td> Value min <input type='text' value='-1000' name='ValeurMini' size=8 maxlength=10>\n"
    " max <input type='text' value='1000' name='ValeurMaxi' size=8 maxlength=10>\n"
    " Time base <select name='TempsC'>\n";
static char menu4b[] =    
    "</select> mS\n"
    "<input type='submit' value='OK' name='ChoixC' onClick='sauve()'></td></tr>\n"
    "<tr><td bgcolor=violet>\n"
    "<select name='CMem2'>\n";
static char menu5[] =
    "</select>\n Address $<input type='text' name='IndexLect2' size=8 maxlength=8>\n"
    " Magenta curve <input type='checkbox' value='OK' name='SelC2'></td>\n"
    "<td></td></tr>\n"
    "<tr><td bgcolor=deepskyblue>\n"
    "<select name='CMem3'>\n";
static char menu6[] =
    "</select>\n Address $<input type='text' name='IndexLect3' size=8 maxlength=8>\n"
    " Blue curve &nbsp;&nbsp;&nbsp;&nbsp;&nbsp; <input type='checkbox' value='OK' name='SelC3'></td>\n"
    "<td></td></tr>\n"
    "<tr><td bgcolor=cyan>\n"
    "<select name='CMem4'>\n";
static char menu7[] =
    "</select>\n Address $<input type='text' name='IndexLect4' size=8 maxlength=8>\n"
    " Cyan curve &nbsp;&nbsp;&nbsp;&nbsp; <input type='checkbox' value='OK' name='SelC4'></td>\n"
    "<td></td></tr>\n"
    "<tr><td>\n"
    "Trigger on <select name='TrigMem'>\n";
static char menu8[] =
    "</select>\n Address $<input type='text' value='0' name='IndexTrig' size=8 maxlength=8></td>\n"
    "<td> Level <input type='radio' value='Bas' name='SignTrig'> Low\n"
    "<input type='radio' value='Haut' name='SignTrig' checked> High\n"
    "<input type='text' value='1' name='ValeurTrig' size=8 maxlength=10>\n"
    " Pretrigger delay <input type='int' value='50' name='PreTrig' size=2 maxlength=2> \%\n"
    "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; "
    "<input type='submit' value='Look' name='VoirC' onClick='sauve()'>\n"
    "</td></tr></table>\n"
   "<script language='javascript'>\n"
    "<!-- \n"
    "  var message3 = \"Allows to see curves until 4 choices of variables, at the difference with the other pages, \" \n"
    "   + \"the 4 choices are global i.e. another window opened with others variables overwrite the settings \" \n"
    "   + \"of the first window. \"; \n"
    "  if(BulleBill) document.write(\"<span title='\" + message3 + \"'>\"); \n"
    "  else document.write(\"<span>\"); \n"
    "// --> \n"
    "</script>\n"
    "<b onMouseOver='this.style.color=\"blue\"; bulle(message3 ,event);' onMouseOut='this.style.color=\"black\"; fin_bulle();'> \n"
    "Curve from variable : </b></span><br clear=right>\n"
#endif /* WEB_LIGHT */
#ifdef DBUG
    "<p><script language='javascript'>\n"
    "<!-- \n"
    "  var message4 = \"Allows to disassemble memory. \"; \n"
    "  if(BulleBill) document.write(\"<span title='\" + message4 + \"'>\"); \n"
    "  else document.write(\"<span>\"); \n"
    "// --> \n"
    "</script>\n"
    "<b onMouseOver='this.style.color=\"blue\"; bulle(message4, event);' onMouseOut='this.style.color=\"black\"; fin_bulle();'> \n"
    "Disassemble memory : </b></span>\n"
    " Start address $<input type='text' value='e00000' name='OffsetDis' size=8 maxlength=8>\n"
    " Length <input type='int' value='32' name='LengthDis' size=3 maxlength=3>\n"
     "<input type='submit' value='OK' name='ChoixDis' onClick='sauve()'>\n"
#endif
    "<p><script language='javascript'>\n"
    "<!-- \n"
    "  var message5 = \"Access to the upload program page in srecord format (.HEX), however the unknow files are copied inside the ram disk of 8 MB. \"; \n"
    "  if(BulleBill) document.write(\"<span title='\" + message5 + \"'>\"); \n"
    "  else document.write(\"<span>\"); \n"
    "// --> \n"
    "</script>\n"
    "<b onMouseOver='this.style.color=\"blue\"; bulle(message5 ,event);' onMouseOut='this.style.color=\"black\"; fin_bulle();'> \n"
    "Upload : </b></span><input type='submit' value='File' name='EnvoiFichier' onClick='sauve()'>\n"
    "<p><script language='javascript'>\n"
    "<!-- \n"
    "  var message6 = \"Start a Telnet connection for get the shell CMD. \"; \n"
    "  if(BulleBill) document.write(\"<span title='\" + message6 + \"'>\"); \n"
    "  else document.write(\"<span>\"); \n"
    "// --> \n"
    "</script>\n"
    "<b onMouseOver='this.style.color=\"blue\"; bulle(message6 ,event);' onMouseOut='this.style.color=\"black\"; fin_bulle();'> \n"
    "Commands shell : </b></span><input type='button' value='CMD via Telnet' onClick='location=\"telnet://\" + document.location.host'>\n"
//    "<input type='submit' value='Internal CMD' name='Telnet'>\n"
    "<p><hr size=1 color=blue>\n"
    "</form>\n";
static char menu_end[] = "</body></html>";
static char mem_html[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB</title>\n"
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE</title>\n"
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE</title>\n"
#endif
#endif
    "</head>\n";
#if 0
static char mem_java[] = 
    "<script language='javascript'>\n"
    "<!-- \n"
    "  document.write(\"<applet codebase='.' archive='jta26.jar' code='de.mud.jta.Applet' width=590 height=360>\")\n"
    "  document.write(\"<param name='Socket.host' value='\" + document.location.host + \"'>\")\n"
    "  document.write(\"<h1>Your browser not support Java applets!</h1>\")\n"
    "  document.write(\"</applet>\")\n"
    "// --> \n"
    "</script>\n";
#endif
static char mem_refresh[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB - Mem</title>\n"
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE - Mem</title>\n"
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE - Mem</title>\n"
#endif
#endif
#if 0
    "<script language='javascript'>\n"
    "<!-- \n"
    "function exportToXL(eSrc) { \n"
    "  var oExcel; \n"
    "  var oExcelSheet; \n"
    "  var oWkBooks; \n"
    "  var cols; \n"
    "  oExcel = new ActiveXObject('Excel.Application'); \n"
    "  oWkBooks = oExcel.Workbooks.Add; \n"
    "  oExcelSheet = oWkBooks.Worksheets(1); \n"
    "  oExcelSheet.Activate(); \n"
    "  if(eSrc.tagName != 'TABLE') \n"
    "    return false; \n"
    "  cols = Math.ceil(eSrc.cells.length / eSrc.rows.length); \n"
    "  for(var i = 0; i < eSrc.cells.length; i ++) { \n"
    "    var c, r; \n"
    "    r = Math.ceil((i+1) / cols); \n"
    "    c = (i+1)-((r-1)*cols) \n"
    "    if(eSrc.cells(i).tagName == 'TH') { \n"
    "      oExcel.ActiveSheet.Cells(r,c).Font.Bold = true; \n"
    "      oExcel.ActiveSheet.Cells(r,c).Interior.Color = 14474460; } \n"
    "    if(eSrc.cells(i).childNodes.length > 0 && eSrc.cells(i).childNodes(0).tagName == \"B\") \n"
    "      oExcel.ActiveSheet.Cells(r,c).Font.Bold = true; \n"
    "    oExcel.ActiveSheet.Cells(r,c).Value = eSrc.cells(i).innerText; } \n"
    "  oExcelSheet.Application.Visible = true; } \n"
    "// --> \n"
    "</script>\n"
#endif
    "<meta http-equiv='refresh' content='";
#ifndef WEB_LIGHT
static char mem_redirection_curve[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB</title>\n"
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE</title>\n"
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE</title>\n"
#endif
#endif
    "<meta http-equiv='refresh' content='0; url=/curve.html'>\n"
    "</head>\n"
    "<body>\n<a href='/curve.html'></a>\n";
#endif
static char mem_redirection_file[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB</title>\n"
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE</title>\n"
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE</title>\n"
#endif
#endif
    "<meta http-equiv='refresh' content='0; url=/sel_file.html'>\n"
    "</head>\n"
    "<body>\n<a href='/sel_file.html'></a>\n";
static char mem_print[] = 
    "<form><p align=right>\n"
    "<input type='button' value='MENU' name='MENU' onClick='document.location.href=\"/menu.html\"'>\n"
//    "<input type='button' value='EXCEL' name='EXCEL' onCLick='exportToXL(oForm.all(\"idTable\"))'>\n"
    "<input type='button' value='PRINT' name='IMPRIMER' onClick='window.print()'>\n"
    "</p></form>\n";   
static char mem_print2[] = 
    "<form><p align=right>\n"
    "<input type='button' value='MENU' name='MENU' onClick='document.location.href=\"/menu.html\"'>\n"
    "<input type='button' value='PRINT' name='IMPRIMER' onClick='window.print()'>\n"
    "</p></form>\n";   
static char mem_back[] =
    "<form>\n"
    "<input type='button' value='BACK' name='RETOUR' onClick='history.back()'>\n"
    "</form>\n";  
static char mem_end[] = "</body></html>";
static char screen_html[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB - Screen</title>\n"
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE - Screen</title>\n"
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE - Screen</title>\n"
#endif
#endif
    "<script language='javascript'>\n"
    "<!-- \n"
    "var IE = false; \n"
    "if(navigator.appName == \"Microsoft Internet Explorer\") \n"
    "  IE = true; \n"
    "if(!IE) document.captureEvents(Event.MOUSEMOVE) \n"
    "document.onmousemove = getMouseXY; \n"
    "var tempX = 0; \n"
    "var tempY = 0; \n"
    "function getMouseXY(e) { \n"
    "  if(IE) { \n"
    "    tempX = event.clientX + document.body.scrollLeft; \n"
    "    tempY = event.clientY + document.body.scrollTop; \n"
    "  } else { \n"
    "    tempX = e.pageX; \n"
    "    tempY = e.pageY; } \n"
    "  if(document.images && document.images.screen) { \n"
    "    tempX = tempX - GetOffset(document.getElementById('screen'),'offsetLeft'); \n"
    "    tempY = tempY - GetOffset(document.getElementById('screen'),'offsetTop'); \n"
    "    if(tempX >= document.images.screen.width) tempX = -1; \n"
    "    if(tempY >= document.images.screen.height) tempY = -1; \n"
    "  } \n"
    "  if(tempX < 0) tempX = -1; \n"
    "  if(tempY < 0) tempY = -1; \n" 
    "  document.forms[0].MouseX.value = tempX; \n"
    "  document.forms[0].MouseY.value = tempY; \n"
    "  return true; } \n"
    "function GetOffset(inOBJ, inOFF) { \n"
    "  var iVal = 0, oObj = inOBJ; \n"
    "  while(oObj && oObj.tagName != 'BODY') { \n"
    "    iVal += eval('oObj.' + inOFF); \n"
    "    oObj = oObj.offsetParent; } \n"
    "  return iVal; } \n"
    "function detectKey(e) { \n"
    "  var characterCode; \n"
    "  if(e && e.which) characterCode = e.which; \n"
    "  else { \n"
    "    e = event \n"
    "    characterCode = e.keyCode; } \n" /* IE */
    "  if((document.forms[0].MouseX.value >= 0) && (document.forms[0].MouseY.value >= 0)) httpRequest('%' + characterCode); \n"
    "  else if(characterCode == 13) { httpRequest(''); return true; } \n"
    "  return false; } \n"
    "function getMouseState(e) { \n"
    "  if(e && e.which) document.forms[0].MouseK.value = e.which; \n"
    "  else document.forms[0].MouseK.value = 1; \n"
    "  httpRequest(''); } \n"
    "function reload() { \n"
    "  document.images.screen.src = '/gif/screen.gif?' + new Date().getTime();  \n"
    "  if((document.forms[0].MouseX.value >= 0) && (document.forms[0].MouseY.value >= 0)) httpRequest(''); \n"
    "  setTimeout('reload()',500); } \n"
    "function httpRequest(key) { \n"
    "  var xhr_object = null; \n"
    "  var f = document.forms[0]; \n"
    "  if(!document.all && window.XMLHttpRequest) xhr_object = new XMLHttpRequest(); \n"
    "  else if(window.ActiveXObject) { \n"
    "    try { xhr_object = new ActiveXObject('Microsoft.XMLHTTP'); } \n"
    "    catch(e) { alert('Your browser not support XMLHTTPRequest!'); return; } \n"
    "  } \n"
    "  else if(document.all && window.XMLHttpRequest) xhr_object = new XMLHttpRequest(); \n"
    "  else { alert('Your browser not support XMLHTTPRequest!'); return; } \n"
    "  try { xhr_object.open('POST', '/change_screen.html', true); } \n"
    "  catch(e) { return; } \n"
    "  xhr_object.onreadystatechange = function() { \n"
    "    if(xhr_object.readyState == 4) { delete xhr_object.onreadystatechange; xhr_object = null; } \n"        
    "  } \n"
    "  xhr_object.setRequestHeader('Content-type', 'application/x-www-form-urlencoded'); \n"
    "  xhr_object.send('?Key=' + key + '&MouseX=' + f.MouseX.value + '&MouseY=' + f.MouseY.value + '&MouseK=' + f.MouseK.value + '&OffsetX=' + f.OffsetX.value + '&OffsetY=' + f.OffsetY.value + ' '); } \n"
    "// --> \n"
    "</script>\n"
    "</head>\n"
    "<body style='background-color:black; color:white;' onLoad='reload()' onKeyPress='detectKey(event)'>\n"
    "<p><center>\n"
    "<form>\n"
    "<input type='hidden' value='-1' name='MouseX' id='MouseX'>\n"
    "<input type='hidden' value='-1' name='MouseY' id='MouseY'>\n"
    "<input type='hidden' value='0' name='MouseK' id='MouseK'>\n";
static char end_screen_html[] =    
    "</form>\n"
    "<p><div onMousedown='getMouseState(event)' onMouseup='document.forms[0].MouseK.value=0; httpRequest(\"\")'>\n"
    "<img id='screen' name='screen' src='/gif/screen.gif' width=640 height=480>\n"
    "</div>\n"
    "</body></html>";
#ifndef WEB_LIGHT
static char curve_html[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB - Curve</title>\n";
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE - Curve</title>\n";
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE - Curve</title>\n";
#endif
#endif
static char curve_refresh[] = 
    "<meta http-equiv='refresh' content='3'>\n";
static char curve_script[] =
    "<script language='javascript'>\n"
    "<!-- \n"
    "function calc_min_max() { \n"
    "  var pts,i; \n"
    "  if(document.forms[0].Echelle.selectedIndex < 0) \n"
    "    document.forms[0].Echelle.selectedIndex = 0; \n"
    "  if(document.forms[0].Echelle.selectedIndex >= document.forms[0].Echelle.length) \n"
    "    document.forms[0].Echelle.selectedIndex = document.forms[0].Echelle.length-1; \n"
    "  pts = parseFloat(document.forms[0].Echelle.options[document.forms[0].Echelle.selectedIndex].text) * ";
static char curve_end_script[] =
    "  for(i=0;i<document.forms[0].ValeurMini.length;i++) { \n"
    "    if(parseFloat(document.forms[0].ValeurMini.options[i].text) == -pts) \n"
    "      document.forms[0].ValeurMini.selectedIndex=i; } \n"
    "  for(i=0;i<document.forms[0].ValeurMaxi.length;i++) { \n"
    "    if(parseFloat(document.forms[0].ValeurMaxi.options[i].text) == pts) \n"
    "      document.forms[0].ValeurMaxi.selectedIndex=i; } } \n"
    "// --> \n"
    "</script>\n";
static char curve_end_head[] =
    "</head>\n"
    "<body bgcolor=white>\n"
    "<p><center><h3>Curve of ";
//static char curve_file[] = 
//    "<img src='/gif/curve.gif' name='courbe' width=640 height=480>\n";
static char curve_file_redraw[] = 
    "<p><script language='javascript'>\n"
    "<!-- \n"
    "document.write(\"<img src='/gif/curve.gif?\" + new Date().getTime() + \"' name='courbe' width=640 height=480>\");\n"
    "// --> \n"
    "</script>\n";
static char curve_boutons[] = 
    "<form action='/curve.html' method='get' name='submitform'><p align=center>\n";
static char curve_print[] = 
    "<input type='button' value='MENU' name='MENU' onClick='document.location.href=\"/menu.html\"'>\n"
    "<input type='button' value='PRINT' name='IMPRIMER' onClick='window.print()'>\n"
    "</p></form>\n";
static char curve_end[] = "</body></html>";
#endif /* WEB_LIGHT */
static char select_html[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB - File</title>\n"
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE - File</title>\n"
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE - File</title>\n"
#endif
#endif
    "</head>\n";
static char select_form[] =
    "<body bgcolor=#FFFF7F'>\n"
    "<p><center><font color=blue size=5><i><b>"
    "Upload file / system"
    "</b></i></font><br><hr size=1 color=blue><br></center>\n"
    "<p><form action='/file.html' enctype='multipart/form-data' method='post'>\n"
    "<p>Send file : \n"
    "<input type='hidden' value='' name='CheminFichier' id='CheminFichier'>\n"
    "<input type='file' value='' name='Fichier' size=60>\n"
    "<input type='button' value='Send' name='EnvoiFichier' onClick='document.forms[0].CheminFichier.value = document.forms[0].Fichier.value; submit();'>\n"
    "<input type='button' value='MENU' name='MENU' onClick='document.location.href=\"/menu.html\"'>\n"
    "</form>\n";
static char select_end[] = "</body></html>";    
    
static char file_html[] =
#ifdef MCF5445X
    "<html><head>\n<title>M54455EVB - End File Transfer</title>\n";
#else
#ifdef MCF547X
    "<html><head>\n<title>FIREBEE - End File Transfer</title>\n";
#else /* MCF548X */
    "<html><head>\n<title>M5484LITE - End File Transfer</title>\n";
#endif
#endif
static char file_back[] = 
    "<p><form>\n"
    "<input type='button' value='BACK' name='RETOUR' onClick='history.back()'>\n"
    "</form>\n";   
static char file_end[] = "</body></html>";

static xQueueHandle mbox; 

#ifndef WEB_LIGHT
static unsigned long HeureMes,HeureTrig;
static float *Pts, *latch_Pts;
static int PosBuf[NB_COURBES];
static int CMem[NB_COURBES];
static int IndexLect[NB_COURBES];
static float ValeurMini, ValeurMaxi, ValeurTrig;
static int TempsC, TrigMem, IndexTrig, SignTrig, PreTrig, OffsetPos, OffsetTrig;
static float *table_min_max;
#endif

static int Flash, Error;
static unsigned char rec_len;  /* longueur S-record  */
static unsigned char CS;       /* checksum */
static char sbuf[256];
static char *sbuf_ptr;
static unsigned long lowest_address, hight_address, Mem_Data;

static int UseCurve;
static char *BufPic, *BufGif, *SecondBufGif;
int SizeGif;
static int MouseX, MouseY, MouseK, OffsetX, OffsetY;

extern struct fb_info *info_fvdi;
extern long get_videl_base(void);
extern long get_videl_bpp(void);
extern long get_videl_width(void);
extern long get_videl_height(void);
extern long get_videl_size(void);
extern void *get_videl_palette(void);

#ifndef WEB_LIGHT

static float def_table_min_max[] = 
{
  -1000.0, // mini page menu
  -5000000.0, -2000000.0, -1000000.0,
  -500000.0, -200000.0, -100000.0,
  -50000.0, -20000.0, -10000.0,
  -5000.0, -2000.0, -1000.0,    
  -500.0, -200.0, -100.0,
  -50.0, -20.0, -10.0,
  -5.0, -2.0, -1.0,
  -0.5, -0.2, -0.1,
  0.0,
  0.1, 0.2, 0.5,
  1.0, 2.0, 5.0,
  10.0, 20.0, 50.0,
  100.0, 200.0, 500.0,
  1000.0, 2000.0, 5000.0,
  10000.0, 20000.0, 50000.0,
  100000.0, 200000.0, 500000.0,
  1000000.0, 2000000.0, 5000000.0,
  1000.0 // maxi page menu
};

static float table_echelle[] = 
{
  0.2/NB_LIGNES, 0.4/NB_LIGNES, 1.0/NB_LIGNES,
  2.0/NB_LIGNES, 4.0/NB_LIGNES, 10.0/NB_LIGNES,
  20.0/NB_LIGNES, 40.0/NB_LIGNES, 100.0/NB_LIGNES,
  200.0/NB_LIGNES, 400.0/NB_LIGNES, 1000.0/NB_LIGNES,
  2000.0/NB_LIGNES, 4000.0/NB_LIGNES, 10000.0/NB_LIGNES,
  20000.0/NB_LIGNES, 40000.0/NB_LIGNES, 100000.0/NB_LIGNES,
  200000.0/NB_LIGNES, 400000.0/NB_LIGNES, 1000000.0/NB_LIGNES,
  2000000.0/NB_LIGNES, 4000000.0/NB_LIGNES, 10000000.0/NB_LIGNES
};

#endif

static http_status m_http_status10[] = {
  { 200, "HTTP/1.0 200 OK\r\n" },
  { 201, "HTTP/1.0 201 Created\r\n" },
  { 202, "HTTP/1.0 202 Accepted\r\n" },
  { 204, "HTTP/1.0 204 No Content\r\n" },
  { 205, "HTTP/1.0 205 Reset Content\r\n" },
  { 206, "HTTP/1.0 206 Partial Content\r\n" },
  { 301, "HTTP/1.0 301 Moved Permanently\r\n" },
  { 302, "HTTP/1.0 302 Found\r\n" },
  { 304, "HTTP/1.0 304 Not Modified\r\n" },
  { 400, "HTTP/1.0 400 Bad Request\r\n" },
  { 401, "HTTP/1.0 401 Unauthorized\r\n" },
  { 403, "HTTP/1.0 403 Forbidden\r\n" },
  { 404, "HTTP/1.0 404 Not Found\r\n" },
  { 500, "HTTP/1.0 500 Internal Server Error\r\n" },
  { 501, "HTTP/1.0 501 Not Implemented\r\n" },
  { 502, "HTTP/1.0 502 Bad Gateway\r\n" },
  { 503, "HTTP/1.0 503 Service Unavailable\r\n" },
};

static http_status m_http_status11[] = {
  { 100, "HTTP/1.1 100 Continue\r\n" },
  { 101, "HTTP/1.1 101 Switching Protocols\r\n" },
  { 200, "HTTP/1.1 200 OK\r\n" },
  { 201, "HTTP/1.1 201 Created\r\n" },
  { 202, "HTTP/1.1 202 Accepted\r\n" },
  { 203, "HTTP/1.1 203 Non-Authorative Information\r\n" },
  { 204, "HTTP/1.1 204 No Content\r\n" },
  { 205, "HTTP/1.1 205 Reset Content\r\n" },
  { 206, "HTTP/1.1 206 Partial Content\r\n" },
  { 300, "HTTP/1.1 300 Multiple Choices\r\n" },
  { 301, "HTTP/1.1 301 Moved Permanently\r\n" },
  { 302, "HTTP/1.1 302 Found\r\n" },
  { 303, "HTTP/1.1 303 See Other\r\n" },
  { 304, "HTTP/1.1 304 Not Modified\r\n" },
  { 305, "HTTP/1.1 305 Use Proxy\r\n" },
  { 307, "HTTP/1.1 307 Temporary Redirect\r\n" },
  { 400, "HTTP/1.1 400 Bad Request\r\n" },
  { 401, "HTTP/1.1 401 Unauthorized\r\n" },
  { 402, "HTTP/1.1 402 Payment Required\r\n" },
  { 403, "HTTP/1.1 403 Forbidden\r\n" },
  { 404, "HTTP/1.1 404 Not Found\r\n" },
  { 405, "HTTP/1.1 405 Method Not Allowed\r\n" },
  { 406, "HTTP/1.1 406 Not Acceptable\r\n" },
  { 407, "HTTP/1.1 407 Proxy Authentication Required\r\n" },
  { 408, "HTTP/1.1 408 Request Time-out\r\n" },
  { 409, "HTTP/1.1 409 Conflict\r\n" },
  { 410, "HTTP/1.1 410 Gone\r\n" },
  { 411, "HTTP/1.1 411 Length Required\r\n" },
  { 412, "HTTP/1.1 412 Precondition Failed\r\n" },
  { 413, "HTTP/1.1 413 Request Entity Too Large\r\n" },
  { 414, "HTTP/1.1 414 Request-URI Too Large\r\n" },
  { 415, "HTTP/1.1 415 Unsupported Media Type\r\n" },
  { 416, "HTTP/1.1 416 Requested range not satisfiable\r\n" },
  { 417, "HTTP/1.1 417 Expectation Failed\r\n" },
  { 500, "HTTP/1.1 500 Internal Server Error\r\n" },
  { 501, "HTTP/1.1 501 Not Implemented\r\n" },
  { 502, "HTTP/1.1 502 Bad Gateway\r\n" },
  { 503, "HTTP/1.1 503 Service Unavailable\r\n" },
  { 504, "HTTP/1.1 504 Gateway time-out\r\n" },
  { 505, "HTTP/1.1 505 HTTP Version not supported\r\n" },
};

static void minus(char *s)
{
  while(*s)
  {
    if(*s>='A' && *s<='Z')
      *s += 0x20;
    if(*s=='\\')
      *s='/';
    s++;
  }
}

static int read_socket(sock_conn *conn, char *request, int size, int *size_header)
{
  int i, n = 0, len = 0;
  fd_set data_read;
  struct timeval tv;
  char *ptr = request, *ptr2;
  if(size_header != NULL)  
    *size_header = 0;
  if(conn->errno)
    return(-1);
  do
  {
    FD_ZERO(&data_read);
    FD_SET(conn->sock, &data_read); 
    tv.tv_sec = TIMEOUT;
    tv.tv_usec = 0;
    i = select(FD_SETSIZE, &data_read, NULL, NULL, &tv);
    if((i > 0) & (FD_ISSET(conn->sock, &data_read) != 0))
    {
      n = recv(conn->sock, ptr, size - len, 0);
      if(n >= 0)
      {
        len += n;
        ptr += n;
        *ptr = 0;
        if((size_header != NULL)
         && ((ptr2 = strstr(request, "\r\n\r\n")) != NULL))
        {
          *size_header = (int)(ptr2 - request) + 4;
          break;
        }
      }
      else
        conn->errno = errno;
    }
  }
  while((n >= 0) && (n < size) && (i > 0));
  return((n < 0) ? n : len);
}

static int write_socket(sock_conn *conn, char *response, int size, int flush)
{
  int i, n = 0, len;
  fd_set data_write;
  struct timeval tv;
  char *ptr = response;
  if((conn->buf_write == NULL) || (response == NULL) || conn->errno)
    return(-1);
  do
  {
    len = WRITEBUF_SIZE - conn->offset_buf_write;
    if(size < len)
      len = size;
    if(len >= 0)
    {
      memcpy(&conn->buf_write[conn->offset_buf_write], response, len);
      size -= len;
      conn->offset_buf_write += len;
      response += len;  
    }
    if((conn->offset_buf_write >= WRITEBUF_SIZE)
     || ((size <= 0) && flush))
    {
      len = n = 0;
      ptr = conn->buf_write;
      do
      {
        FD_ZERO(&data_write);
        FD_SET(conn->sock, &data_write);
        tv.tv_sec = TIMEOUT;
        tv.tv_usec = 0;
        i = select(FD_SETSIZE, NULL, &data_write, NULL, &tv);
        if((i > 0) & (FD_ISSET(conn->sock, &data_write) != 0))
        {
          n = send(conn->sock, ptr, conn->offset_buf_write - len, 0);
          if(n >= 0)
          {
            ptr += n;
            len += n;
          }
          else
            conn->errno = errno;
        }
      }
      while((n >= 0) && (len < conn->offset_buf_write) && (i > 0));
      conn->offset_buf_write = 0;
    }
  }
  while(size > 0);
  return(n < 0 ? n : len);
}

static void write_string(sock_conn *conn, char *text)
{
  write_socket(conn, text, strlen(text), 0);
}

static int send_status_code(sock_conn *conn, int status_code)
{
  int i;
  if(conn->version == VERSION_10)
  {
    for(i = 0; i < sizeof(m_http_status10)/sizeof(m_http_status10[0]); i++)
    {
      if(m_http_status10[i].code == status_code)
      {
        write_socket(conn, (char *)m_http_status10[i].text, strlen((void *)m_http_status10[i].text), status_code == 200 ? 0 : 1);
        return 1;
      }
    }
  }
  else
  {
    for(i = 0; i < sizeof(m_http_status11)/sizeof(m_http_status11[0]); i++)
    {
      if(m_http_status11[i].code == status_code)
      {
        write_socket(conn, (char *)m_http_status11[i].text, strlen((void *)m_http_status11[i].text), status_code == 200 ? 0 : 1);
        return 1;
      }
    }
  }
  /* Unsupported status code? Pretty internal error, isn't it? Send 500 */
  send_status_code(conn, 500);
  return 0;
}

static void send_content_type(sock_conn *conn, const char *content_type)
{
  char buf[64];
  strcpy(buf,  "Content-Type: ");
  strcat(buf, content_type);
  strcat(buf, "\r\n");
  write_string(conn, buf);
}

static int dayofweek(int year,int mon,int mday)
{
  static int doylookup[2][13] = {
   { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
   { 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366 }};
  int doe,isleapyear;
  int era,cent,quad,rest;
  /* break down the year into 400, 100, 4, and 1 year multiples */
  rest = year-1;
  quad = rest/4;
  rest %= 4;
  cent = quad/25;
  quad %= 25;
  era = cent/4;
  cent %= 4;
  /* leap year every 4th year, except every 100th year,
     not including every 400th year. */
  isleapyear = !(year % 4) && ((year % 100) || !(year % 400));
  /* set up doe */
  doe = mday + doylookup[isleapyear][mon - 1];
  doe += era * (400 * 365 + 97);
  doe += cent * (100 * 365 + 24);
  doe += quad * (4 * 365 + 1);
  doe += rest * 365;
  return(doe %7);
}

static unsigned long gettime(void)
{
  return(((unsigned long)Tgetdate() << 16) + ((unsigned long)Tgettime() & 0xffff));
}

static void send_date(sock_conn *conn, const char *name, unsigned long datetime)
{
  static char *days[] = { "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun" };
  static char *months[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
  char buf[64], buf2[10];
  int year, mon, mday;
  year = (int)((datetime >> 25) + 1980UL);        /* year */
  mon = (int)((datetime >> 21) & 0xf);          /* month */
  mday = (int)((datetime >> 16) & 0x1f);        /* day of month */
  strcpy(buf, name);
  strcat(buf, days[dayofweek(year,mon,mday)]);
  strcat(buf, ", ");
  ltoa(buf2, mday + 100, 10);
  strcat(buf, &buf2[1]);
  strcat(buf, " ");
  if(mon < 1)
  	mon = 1;
  strcat(buf, months[mon-1]);
  strcat(buf, " ");
  ltoa(buf2, year, 10);
  strcat(buf, buf2);
  strcat(buf, " ");
  ltoa(buf2, ((datetime >> 11) & 0x1f) + 100, 10); /* hour */
  strcat(buf, &buf2[1]);
  strcat(buf, ":");
  ltoa(buf2, ((datetime >> 5) & 0x3f) + 100, 10);  /* mn */
  strcat(buf, &buf2[1]);
  strcat(buf, ":");
  ltoa(buf2, ((datetime & 0x1f) << 1) + 100, 10);  /* sec */
  strcat(buf, &buf2[1]);
  strcat(buf, " GMT\r\n");  
  write_string(conn, buf);
}

static void send_content_length(sock_conn *conn, int length)
{
  char buf[64], buf2[10];
  strcpy(buf,  "Content-Length: ");
  ltoa(buf2, length, 10);
  strcat(buf, buf2);
  strcat(buf, "\r\n");
  write_string(conn, buf);
}

static int copy_file(char *path, char *buf, long size)
{
  extern long ram_disk_drive;
  char subdir[]="C:\\incoming";
  long handle, err;
  char fname[256];
  char *p = path;
  if(ram_disk_drive >= 1)
    subdir[0] =(char)ram_disk_drive + 'A';
  while(*path)
  {
    if((*path == '\\') || (*path == '/'))
      p = path+1;  
    path++;
  }
  if(Fsfirst(subdir, 0x10) != 0)
    Dcreate(subdir);
  strcpy(fname, subdir);
  strcat(fname, "\\");
  strcat(fname, p);
  if(size < 0)
    return(-1);
  if((err = handle = Fcreate(fname, 0)) >= 0)
  {
    err = Fwrite(handle, size, buf);
    Fclose(handle);
    if(err >= 0)
      err = 0;
  }
  return(err);
}

#ifdef MCF5445X

static int Programmation(unsigned long int begin, unsigned long int end, unsigned long int mem)
{
  if((begin < FLASH_TOS_FIRE_ENGINE) || (begin >= FLASH_TOS_FIRE_ENGINE+SIZE_TOS_FLASH))
    return(-1);
  if((end < FLASH_TOS_FIRE_ENGINE) || (end >= FLASH_TOS_FIRE_ENGINE+SIZE_TOS_FLASH))
    return(-1);  
  UnprotectCSBOOT();
  FlashIdentify();
  UnlockFlashBlockAll();
  if(EraseFlash(begin, end) != SUCCESS)
  {
    ResetFlash();
    ProtectCSBOOT();
    return(-1);
  }
  else
  {
    unsigned long *p =(unsigned long *)mem; 
    for(; begin < end; begin += 4)
    {
      if(ProgFlash(begin, *p++) != SUCCESS)
      {
        ResetFlash();
        ProtectCSBOOT();
        return(-1);
      }
    }
  }
  ResetFlash();
  ProtectCSBOOT();
  return(0);
}

#else

#ifdef MCF547X

static int Programmation(unsigned long int begin, unsigned long int end, unsigned long int mem)
{
  if((begin < FLASH_TOS_FIRE_ENGINE) || (begin >= FLASH_TOS_FIRE_ENGINE+SIZE_TOS_FLASH))
    return(-1);
  if((end < FLASH_TOS_FIRE_ENGINE) || (end >= FLASH_TOS_FIRE_ENGINE+SIZE_TOS_FLASH))
    return(-1);  
  UnprotectCSBOOT();
  if(FlashIdentify() != SUCCESS)
  {
    ProtectCSBOOT();
    return(-1);
  }
  if(EraseFlash(begin, end) != SUCCESS)
  {
    ProtectCSBOOT();
    return(-1);
  }
  else
  {
    if(ProgFlash(begin, end, (void *)mem) != SUCCESS)
    {
      ProtectCSBOOT();
      return(-1);
    }
  }
  ProtectCSBOOT();
  return(0);
}

#else /* MCF548X */

static int Programmation(unsigned long int begin, unsigned long int end, unsigned long int mem)
{
  int bytes = (int)(end-begin);
  if((begin < FLASH_TOS_FIRE_ENGINE) || (begin >= FLASH_TOS_FIRE_ENGINE+SIZE_TOS_FLASH))
    return(-1);
  if((end < FLASH_TOS_FIRE_ENGINE) || (end >= FLASH_TOS_FIRE_ENGINE+SIZE_TOS_FLASH))
    return(-1);  
  if(begin >= end)
    return(-1);
  if(begin & INTEL_C3_FLASH_CELL_MASK)
    return(-1);
  bytes += INTEL_C3_FLASH_CELL_MASK;
  bytes &= ~INTEL_C3_FLASH_CELL_MASK;
  intel_c3_flash_init(BOOT_FLASH_BASE);
  if(intel_c3_flash_program(begin, mem, bytes, 1, NULL, NULL) != bytes)
    return(-1);
  return(0);  
}

#endif /* MCF547X */
#endif /* MCF5445X */

/***********************************************************************/
/* getbh: Read 2 ASCII characters from the S-record buffer and         */
/*        convert them into their corresponding binary value           */
/*                                                                     */
/*  INPUT: None                                                        */
/* OUTPUT: value:    Binary value of byte pair                         */
/*         sbuf_ptr: Incremented by 2                                  */
/*                                                                     */
/***********************************************************************/
static unsigned char getbh(void)
{
  unsigned long c, i, value = 0;
  for(i = 0; i < 2; i++)
  {
    c = *sbuf_ptr++;
    value <<= 4;
    value  += c;
    if((c >= '0') && (c <= '9'))
      value -= '0';
    else
    {
      if((c >= 'A') && (c <= 'F'))
        value -= ('A' - 10);
      else
      {
        Error=1;
        return(0);
      }
    }
  }
  return value;
}

/***********************************************************************/
/* getcsbh: Read a byte pair from the record buffer, updating          */
/*          the calculated checksum and remaining record length        */
/*                                                                     */
/*  INPUT: None                                                        */
/* OUTPUT: val: Binary value of byte pair                              */
/*                                                                     */
/***********************************************************************/
static unsigned char getcsbh(void)
{
  unsigned char val;
  CS += (val = getbh());
  return val;
}

/***********************************************************************/
/* read_rec: Read an S-record from the S-record buffer and write       */
/*           the data it represents to memory                          */
/*                                                                     */
/*  INPUT: None                                                        */
/* OUTPUT: None                                                        */
/*                                                                     */
/***********************************************************************/
static void read_rec(void)
{
  unsigned long rec_type, addr, i;
  unsigned char data;
  int bypass_data;
  /*---------------------------------------------------------------------*/
  /* Position sbuf_ptr to just after 'S' of S-record.                    */
  /*---------------------------------------------------------------------*/
  for(sbuf_ptr = sbuf; *sbuf_ptr != 'S';)
  if(!(*++sbuf_ptr))
    return;
  sbuf_ptr++;
  /*---------------------------------------------------------------------*/
  /* Get the record type and lenght of the record. Adjust the checksum   */
  /* byte accordingly.                                                   */
  /*---------------------------------------------------------------------*/
  rec_type = *sbuf_ptr++ - '0'; 
  CS = (rec_len = getbh());
  rec_len--;   
  switch(rec_type)
  {
    case 7:
    case 8:
    case 9:
      /*-------------------------------------------------------------*/
      /* Termination records.  Nothing to do.                        */
      /*-------------------------------------------------------------*/
      i = 11 - rec_type;
      addr = 0;
      while(i--)
      {
        addr = (addr << 8) + getcsbh();
        if(Error)
          return;
      }
      break;
    case 0:
      break;
    case 5:
      /*-------------------------------------------------------------*/
      /* All of these records can be skipped                         */
      /*-------------------------------------------------------------*/
      while(rec_len != 0)
      {
        getcsbh();
        rec_len--;
        if(Error)
          return;
      }
      break;
    case 1:
    case 2:
    case 3:
      /*-------------------------------------------------------------*/
      /* Data record - first read the load address from the record   */
      /* and add in the offset.  The load address is 2, 3, or 4 byte */
      /* pairs, depending on the record type.                        */
      /*-------------------------------------------------------------*/
      addr = 0;
      for(i = 0; i <= rec_type; i++)
      {
        addr = (addr << 8) + getcsbh();
        rec_len--;
      }
      bypass_data = 0;
      if(addr < lowest_address)
      {
        if(!Flash || (Flash && (addr >= FLASH_TOS_FIRE_ENGINE)))
          lowest_address = addr;
        else
          bypass_data = 1;
      }
      if((addr + rec_len) > hight_address)
        hight_address = addr + rec_len;
      if(Error)
        return;
      /* controle si le telechargement est en flash */
      if((addr >= FLASH_TOS_FIRE_ENGINE)
       && (addr < FLASH_TOS_FIRE_ENGINE+SIZE_TOS_FLASH))
        Flash=1;
      addr &= (SIZE_TOS_FLASH-1);
      addr += Mem_Data;
      /*-------------------------------------------------------------*/
      /* Read each byte of data.                                     */
      /*-------------------------------------------------------------*/
      while(rec_len != 0)
      {
        data = getcsbh();
        if(!bypass_data)
          *((unsigned char *)addr++) = data;
        rec_len--;
        if(Error)
          return;
      }
      break;
    default:
      Error = 1;
      break;
  }
  /*---------------------------------------------------------------------*/
  /* Last thing to do is verify the checksum.                            */
  /*---------------------------------------------------------------------*/
  if((rec_type != 0) && (getbh() != (unsigned char)(~CS)))
    Error = 1;
}

/***********************************************************************/
/* conv_rec: Process a buffer full of records                          */
/*                                                                     */
/*  INPUT: buf_ptr: pointer to buffer                                  */
/*         count:   length of buffer                                   */
/* OUTPUT: None                                                        */
/*                                                                     */
/***********************************************************************/
static void conv_rec(char *buf_ptr, long count)
{
  unsigned char c;
/*---------------------------------------------------------------------*/
/* Transfer data from the buffer into the record buffer until a CR,    */
/* NL, or NULL is transferred (any of these is assumed to terminate    */
/* a record).  Then call read_rec() to process the record, and start   */
/* filling the record buffer again.                                    */
/* NOTE: If the buffer ends with a partial record, it will be saved in */
/*       sbuf[].                                                       */
/*---------------------------------------------------------------------*/
  while(count-- > 0)
  {
    c = (*sbuf_ptr++ = *buf_ptr++);
    if(c == '\n' || c == '\r' || c == 0)
    {
      *sbuf_ptr = '\0';
      read_rec();
      if(Error)
        return;
      while(*buf_ptr == '\n' || *buf_ptr == '\r')
      {
        buf_ptr++;
        if(--count <= 0)
          break;
      }
      sbuf_ptr = sbuf;
    }
  }
}

/***********************************************************************/
/* HTML pages                                                          */
/***********************************************************************/

static void add_block(sock_conn *conn, const char *data, int size)
{
  if(size + conn->response_size > conn->response_max_size)
  {
    void *buf = conn->response;
    int len = conn->response_max_size;
    do
      conn->response_max_size += WRITEBUF_SIZE;
    while(size + conn->response_size > conn->response_max_size);
    conn->response = pvPortMalloc2(conn->response_max_size);
    memcpy(conn->response, buf, len);
    vPortFree2(buf);
  }
  if(conn->response != NULL)
  {
    memcpy(&conn->response[conn->response_size], data, size);
    conn->response_size += size;
  }
}

static void a2p(sock_conn *conn, const char *text)
{
  add_block(conn, text, strlen(text));
}

static void add_types(sock_conn *conn)
{
  int i=0;
  while(types1[i])
  {
    a2p(conn, "<option>");
    a2p(conn, types1[i++]);
    a2p(conn, "</option>\n");
  }
}

static void add_types_write(sock_conn *conn)
{
  int i=0;
  while(types2[i])
  {
    a2p(conn, "<option>");
    a2p(conn, types2[i++]);
    a2p(conn, "</option>\n");
  }
}

#ifndef WEB_LIGHT
static void add_types_curve(sock_conn *conn)
{
  int i=0;
  while(types3[i])
  {
    a2p(conn, "<option>");
    a2p(conn, types3[i++]);
    a2p(conn, "</option>\n");
  }
}
#endif

static void handle_menu(sock_conn *conn, char *request)
{
#ifndef WEB_LIGHT
  unsigned long ms = 1000UL/configTICK_RATE_HZ;
#endif
  char buf[10];
  a2p(conn, menu_html);
  a2p(conn, menu);
  ltoa(buf, VERSION, 16);
  a2p(conn, buf);
  a2p(conn, menu0);
  if(((info_fvdi != NULL) && (info_fvdi->var.bits_per_pixel >= 16))
   || ((info_fvdi == NULL) && get_videl_base() && ((get_videl_bpp() == 4) || (get_videl_bpp() >= 16))))
    a2p(conn, "<p><input type='button' value='View Screen' onClick='document.location.href=\"/screen.html\"'>\n");
  a2p(conn, menu1);
  add_types(conn);
  a2p(conn, menu2);
  add_types_write(conn);
  a2p(conn, menu3);
#ifndef WEB_LIGHT
  add_types_curve(conn);
  a2p(conn, menu4);
  while(ms <= 1000)
  {
    a2p(conn, "<option>");
    ltoa(buf, (long)ms, 10);  
    a2p(conn, buf);
    a2p(conn, "</option>\n");
    switch(ms)
    {
      case 1: ms = 2; break;
      case 2: ms = 5; break;
      case 5: ms = 10; break;
      case 10: ms = 20; break;
      case 20: ms = 50; break;
      case 50: ms = 100; break;
      case 100: ms = 200; break;
      case 200: ms = 500; break;
      case 500: ms = 1000; break;
      case 1000: ms = 2000; break;
    }
  }
  a2p(conn, menu4b);
  add_types_curve(conn);
  a2p(conn, menu5);
  add_types_curve(conn);
  a2p(conn, menu6);
  add_types_curve(conn);
  a2p(conn, menu7);
  a2p(conn, "<option>");
  a2p(conn, "None");
  a2p(conn, "</option>\n");
  add_types_write(conn);
  a2p(conn, menu8);
#endif
  a2p(conn, menu_end);
}

static void handle_mem(sock_conn *conn, char *req)
{
  int Refresh=1;
  int i,j,max_align,max,choix,ChoixMem,OffsetLect=0,LengthLect=1,IndexMem=0;
#ifdef DBUG
  int  OffsetDis=0,LengthDis=8;
#endif
  unsigned long bits;  
  char body[64],buf[BUF_SIZE],buf2[BUF_SIZE];
  char *ptr, *ptr2;
  short *sptr;
#ifdef MCF5445X
  long val;
#else /* MCF548X */
  float val;
  double *dptr;
#endif /* MCF5445X */
#ifndef WEB_LIGHT
  int suppr,k;
#endif
  int len;
  long *mem = NULL;
  ptr = strchr(req, '?');
  if(ptr == NULL)
    ptr = req;
  ptr2 = buf;
  i = 0;
  while(*ptr && (*ptr != ' ') && (i++ < BUF_SIZE-1))
    *ptr2++ = *ptr++;
  *ptr2 = 0;
  if((ptr = strstr(buf, "ColorMem=")) != NULL
   && ptr[9]!='&' && ptr[9])
  {
    buf2[6] = 0;
    memcpy(buf2, &ptr[9], 6);
    strcpy(body, "<body bgcolor=#");
    strcat(body, buf2);
    strcat(body, ">\n<p>\n");
  }  
  else
    strcpy(body, "<body bgcolor=#FFFF7F>\n<p>\n");
/*************************** ENVOI FICHIER ************************************/
  if(strstr(buf, "EnvoiFichier=") != NULL)
    a2p(conn, mem_redirection_file);
#if 0
/******************************* TELNET ***************************************/
  else if(strstr(buf, "Telnet=") != NULL)
  {
    a2p(conn, mem_html);
    a2p(conn, body);
    a2p(conn, mem_java);
    a2p(conn, mem_back);
  }
#endif
/***************************** DISASSEMBLE *************************************/
#ifdef DBUG
  else if(strstr(buf, "ChoixDis=") != NULL)
  {
    struct DisasmPara_68k dp;
    char buffer[16];
    m68k_word *ip, *p;
    char opcode[16];
    char operands[128];
    char iwordbuf[32];
    char *s;
    int n, i;
    if((ptr = strstr(buf, "OffsetDis=")) != NULL
     && ptr[10]!='&')
    {
      ptr[9] = '$';
      OffsetDis = (int)atol(&ptr[9]);
      ptr[9] = '=';
    }
    else
      OffsetDis = 0;
    if((ptr = strstr(buf, "LengthDis=")) != NULL
     && ptr[10]!='&')
    {
      LengthDis = (int)atol(&ptr[10]);
      if(LengthDis < 0)
        LengthDis = 8;
      else if(LengthDis > 999)
        LengthDis = 999;
    }
    else
      LengthDis = 1;
    a2p(conn, mem_html);
    a2p(conn, body);
    a2p(conn, "<pre>\r\n");
    p=(m68k_word *)OffsetDis;
    db_radix = 16;
    dp.instr = NULL;              /* pointer to instruction to disassemble */
    dp.iaddr = NULL;              /* instr.addr., usually the same as instr */
    dp.opcode = opcode;           /* buffer for opcode, min. 16 chars. */
    dp.operands = operands;       /* operand buffer, min. 128 chars. */
    dp.radix = 16;                /* base 2, 8, 10, 16 ... */
/* call-back functions for symbolic debugger support */
    dp.get_areg = NULL;           /* returns current value of reg. An */
    dp.find_symbol = NULL;        /* finds closest symbol to addr and */
                                  /*  returns (positive) difference and name */
/* changed by disassembler: */
    dp.type = 0;                  /* type of instruction, see below */
    dp.flags = 0;                 /* additional flags */
    dp.reserved = 0;
    dp.areg = 0;                  /* address reg. for displacement (PC=-1) */
    dp.displacement = 0;          /* branch- or d16-displacement */
    for(i=0;i<LengthDis;i++)
    {          
      for(n = 0; n<sizeof(opcode)-1; opcode[n++]=' ');
      opcode[n] = 0;
      for(n = 0; n<sizeof(operands); operands[n++]=0);
      dp.instr = dp.iaddr = p;
      p = M68k_Disassemble(&dp);
      /* print up to 5 instruction words */
      for(n = 0; n<26; iwordbuf[n++]=' ');
      iwordbuf[26] = 0;
      if((n = (int)(p-dp.instr)) > 5)
        n = 5;
      ip = dp.instr;
      s = iwordbuf;
      while(n--)
      {
        ltoa(buffer,(((unsigned long)*(unsigned char *)ip)<<8) | ((unsigned long)*((unsigned char *)ip+1)) | 0x10000UL,16);
        *s++ = buffer[1];
        *s++ = buffer[2];
        *s++ = buffer[3];
        *s++ = buffer[4];
        s++;
          ip++;
      }
      a2p(conn, "$");
      ltoa(buffer,(unsigned long)dp.iaddr,16);
      a2p(conn, buffer);
      a2p(conn, ": ");
      a2p(conn, iwordbuf);
      strcpy(buffer, "       ");
      n = 0;
      while(opcode[n])
      {
        buffer[n] = opcode[n];
        n++;
      }
      buffer[n] = 0;
      a2p(conn, buffer);
      a2p(conn, " ");
      a2p(conn, operands);
      a2p(conn, "\r\n");
    }
    a2p(conn, "</pre>\r\n");
    a2p(conn, mem_print);
    goto EndMem;
  }
#endif
/*************************** EVOLUTION VARIABLE (COURBE) **********************/
#ifndef WEB_LIGHT
  else if(strstr(buf, "VoirC=") != NULL)
    a2p(conn, mem_redirection_curve);
  else if(strstr(buf, "ChoixC=") != NULL)
  {
    choix = 0;
    while(types3[choix] != NULL)
    {
      for(k=0; k<4; k++)
      {
        switch(k)
        {
          case 0: strcpy(buf2, "CMem="); break;
          case 1: strcpy(buf2, "CMem2="); break;
          case 2: strcpy(buf2, "CMem3="); break;
          case 3: strcpy(buf2, "CMem4="); break;
        }
        strcat(buf2, types3[choix]);
        if(strstr(buf, buf2) != NULL)
        {
          suppr = 1;     /* suppression */
          switch(k)
          {
            case 0: strcpy(buf2, "SelC=OK"); break;
            case 1: strcpy(buf2, "SelC2=OK"); break;
            case 2: strcpy(buf2, "SelC3=OK"); break;
            case 3: strcpy(buf2, "SelC4=OK"); break;
          }      
          if(strstr(buf, buf2) != NULL)
            suppr = 0;   /* ajout ou modification */
          switch(k)
          {
            case 0: strcpy(buf2, "IndexLect="); break;
            case 1: strcpy(buf2, "IndexLect2="); break;
            case 2: strcpy(buf2, "IndexLect3="); break;
            case 3: strcpy(buf2, "IndexLect4="); break;
          }    
          if((ptr = strstr(buf, buf2)) != NULL
           && (ptr[10]!='&' || suppr))
          {
            ptr[9] = '$';
            i = (int)atol(&ptr[9]);
            ptr[9] = '=';
            if(1 || suppr)
            {
              j = IndexLect[k]; 
              if(suppr)
                IndexLect[k] = -1;                
              else
              IndexLect[k] = i;
              if(IndexLect[k] != j)
              {
                memset(&Pts[k*PICTURE_WIDTH*NB_IMAGES],0,PICTURE_WIDTH*NB_IMAGES*sizeof(float));
                PosBuf[k] = 0;         
              } 
              i = CMem[k];
              if(suppr)
                CMem[k] = -1;
              else
                CMem[k] = choix;
              if(CMem[k] != i)
              {
                memset(&Pts[k*PICTURE_WIDTH*NB_IMAGES],0,PICTURE_WIDTH*NB_IMAGES*sizeof(float));
                PosBuf[k] = 0;  
              }
            }
            else
            {
              a2p(conn, mem_html);
              a2p(conn, "<body bgcolor=red>\n<h3>Bad address, line ");
              buf2[0] = (char)k + '1';
              buf2[1] = '\0';
              a2p(conn, buf2);
              a2p(conn, "</h3>\n"); 
              a2p(conn, mem_back);
              goto EndMem;
            }
          }
          else
          {
            a2p(conn, mem_html);
            a2p(conn, "<body bgcolor=red>\n<h3>No address, line ");
            buf2[0] = (char)k + '1';
            buf2[1] = '\0';
            a2p(conn, buf2);
            a2p(conn, "</h3>\n"); 
            a2p(conn, mem_back);
            goto EndMem;
          }
        }  
      }
      choix++;
    }
    if((ptr = strstr(buf, "ValeurMini=")) != NULL
     && ptr[11]!='&')
      table_min_max[0] = ValeurMini = atof(&ptr[11]); 
    if((ptr = strstr(buf, "ValeurMaxi=")) != NULL
     && ptr[11]!='&')
      table_min_max[(sizeof(def_table_min_max)/sizeof(float))-1] = ValeurMaxi = atof(&ptr[11]);
    if(ValeurMaxi<=ValeurMini)
    {
      val = ValeurMini;
      table_min_max[0] = ValeurMini = ValeurMaxi;
      table_min_max[(sizeof(def_table_min_max)/sizeof(float))-1] = ValeurMaxi = val;
    }
    TempsC = 1000/configTICK_RATE_HZ;
    if((ptr = strstr(buf, "TempsC=")) != NULL
     && ptr[7]!='&')
      TempsC = (int)atol(&ptr[7]);       
    if((strstr(buf, "TrigMem=Aucun") != NULL)
     || (strstr(buf, "TrigMem=None") != NULL))
    {
      if(TrigMem)
      {
        memset(Pts,0,NB_COURBES*PICTURE_WIDTH*NB_IMAGES*sizeof(float));
        for(i=0;i<NB_COURBES;PosBuf[i++]=0);        
      } 
      TrigMem = 0;
      OffsetPos = NB_COLONNES*(NB_IMAGES-1);
    }
    choix = 0;
    while(types2[choix] != NULL)
    {
      strcpy(buf2, "TrigMem=");
      strcat(buf2, types2[choix]);
      if(strstr(buf, buf2) != NULL)
      {
        if((ptr = strstr(buf, "IndexTrig=")) != NULL
         && ptr[10]!='&')
        {
          ptr[9] = '$';
          i = (int)atol(&ptr[9]);
          ptr[9] = '=';
          if(1)
          {
            IndexTrig = i;
            SignTrig = 1;   /* + */
            if(strstr(buf, "SignTrig=Bas") != NULL)
              SignTrig = 0; /* - */
            if((ptr = strstr(buf, "ValeurTrig=")) != NULL
             && ptr[11]!='&')
            ValeurTrig = atof(&ptr[11]);
            if((ptr = strstr(buf, "PreTrig=")) != NULL
             && ptr[8]!='&')
            {
              PreTrig = (int)atol(&ptr[8]);
              if(PreTrig<0 || PreTrig>99)
                PreTrig = 50;
            }
            memset(Pts,0,NB_COURBES*PICTURE_WIDTH*NB_IMAGES*sizeof(float));
            for(i=0;i<NB_COURBES;PosBuf[i++]=0);
            OffsetPos = NB_COLONNES*(NB_IMAGES-1);
            TrigMem = choix + 1;
          }
          else
          {
            a2p(conn, mem_html);
            a2p(conn, "<body bgcolor=red>\n<h3>Bad address of trigger</h3>\n");
            a2p(conn, mem_back);
            goto EndMem;
          }
        }
      }
      choix++;
    }
    a2p(conn, mem_redirection_curve);
  }
#endif
/*************************** MODIFICATION VARIABLE ****************************/
  else if(strstr(buf, "ChoixModif=") != NULL)
  {
    ChoixMem = 0;
    choix = 1;
    while(types2[choix] != NULL)
    {
      strcpy(buf2, "ModifMem=");
      strcat(buf2, types2[choix]);
      len = strlen(buf2);
      if((ptr2 = strstr(buf, buf2)) != NULL
       && (ptr2[len]=='&' || ptr2[len]==0))
      {
        if((ptr = strstr(buf, "IndexModif=")) != NULL
         && ptr[11]!='&')
        {
          ptr[10] = '$';
          mem = (long *)atol(&ptr[10]);
          ptr[10] = '=';
          if((ptr = strstr(buf, "ValeurModif=")) != NULL
           && ptr[12]!='&')
          {
#ifndef MCF5445X
            val = atof(&ptr[12]);
#else
            val = atol(&ptr[12]);
#endif
            if(1)
            {
              if(strcmp(types2[choix], "BYTE") == 0)
                *(char *)mem = (char)val; 
              else if(strcmp(types2[choix], "SHORT") == 0)
                *(short *)mem = (short)val; 
#ifndef MCF5445X
              else if(strcmp(types2[choix], "FLOAT") == 0)
                *(float *)mem = val;
              else if(strcmp(types2[choix], "DOUBLE") == 0)
                *(double *)mem = (double)val;
#endif
              else
                *mem = (long)val;
              ChoixMem = 0;
              while(types1[ChoixMem] != NULL)
              {
                if(strcmp(types1[ChoixMem], types2[choix]) == 0)
                {
                  ChoixMem++;
                  OffsetLect = (int)mem;
                  IndexMem = (int)mem;
                  LengthLect = 16;
                  a2p(conn, mem_html);
                  goto AffMem;
                }
                ChoixMem++;
              }
              ChoixMem = 0;
              OffsetLect = (int)mem;
              IndexMem = (int)mem;
              LengthLect = 16;
              a2p(conn, mem_html);
              goto AffMem;              
            }
            else
            {
              a2p(conn, mem_html);
              a2p(conn, "<body bgcolor=red>\n<h3>Bad address</h3>\n");
              a2p(conn, mem_back);
              goto EndMem;
            }
          }
          else
          {
            a2p(conn, mem_html);
            a2p(conn, "<body bgcolor=red>\n<h3>No value for this address</h3>\n");    
            a2p(conn, mem_back);
            goto EndMem;
          }
        }
        else
        {
          a2p(conn, mem_html);
          a2p(conn, "<body bgcolor=red>\n<h3>No address</h3>\n");
          goto EndMem;
        }
      }
      choix++;
    }
    a2p(conn, mem_html);
    a2p(conn, "<body bgcolor=red>\n<h3>Impossible to change this address</h3>\n");
    a2p(conn, mem_back);
  }
/*************************** LECTURE VARIABLES ********************************/
  else if(strstr(buf, "ChoixLect=") != NULL)
  {
    a2p(conn, mem_refresh);
    if((ptr = strstr(buf, "Rafraichir=")) != NULL
     && ptr[11]!='&')
      Refresh = (int)atol(&ptr[11]);
    if(Refresh == 0)
      Refresh++;
    ltoa(buf2, Refresh, 10);
    a2p(conn, buf2);
    a2p(conn, "'>\n</head>\n");
    if((ptr = strstr(buf, "OffsetLect=")) != NULL
     && ptr[11]!='&')
    {
      ptr[10] = '$';
      OffsetLect = (int)atol(&ptr[10]);
      ptr[10] = '=';
    }
    else
      OffsetLect = 0;
    if((ptr = strstr(buf, "LengthLect=")) != NULL
     && ptr[11]!='&')
    {
      LengthLect = (int)atol(&ptr[11]);
      if(LengthLect < 0)
        LengthLect = 1;
      else if(LengthLect > MAX_LENGTH)
        LengthLect = MAX_LENGTH;
    }
    else
      LengthLect = 1;
    ChoixMem = 0;
AffMem:
    a2p(conn, body);
    mem = (long *)OffsetLect;
    choix = 0;
    while(types1[choix] != NULL)
    {
      strcpy(buf2, "LectMem=");
      strcat(buf2, (ptr = types1[choix]));
      len = strlen(buf2);
      if(((ptr2 = strstr(buf, buf2)) != NULL
       && (ptr2[len]=='&' || ptr2[len]==0)) || ChoixMem)
      {
        if(!ChoixMem || (choix+1)==ChoixMem)
        {
          a2p(conn, "<form id=oForm><table id=idTable name=idTable border width=100%%>\n<caption><h3>Table ");
          a2p(conn, ptr);
          a2p(conn, "</h3></caption>\n");
          if(strcmp(types1[choix], "BIT") == 0)
          {
            for(i=0; i<LengthLect; i++)
            {
              a2p(conn, "<tr>\n");
              bits = (unsigned long)mem[i];
              bits = ((bits & 0xFF)<<24) + ((bits & 0xFF00)<<8)
               + ((bits & 0xFF0000)>>8) + ((bits & 0xFF000000)>>24);
              for(j=0; j<32; j++)
              {
                if(j==16)
                  a2p(conn, "</tr>\n<tr>\n");
                a2p(conn, "<td width=6%% align=center>"); 
                ltoa(buf2, i*32 + j + 1, 10);
                a2p(conn, buf2);
                if(bits&1)
                  a2p(conn, "<br><img src='/gif/green_led.gif' width=10 height=10></td>\n");
                else
                  a2p(conn, "<br><img src='/gif/red_led.gif' width=10 height=10></td>\n");
                bits >>= 1;
              }
              a2p(conn, "</tr>\n");
            }
          }
          else
          {
            max_align = (LengthLect + 15) & ~15;
            max = LengthLect;
            for(i=0; i<max_align; i++)
            {
              if(!(i&15))
              {
                a2p(conn, "<tr>\n<td><b>$");
                if(strcmp(types1[choix], "BYTE") == 0)
                {
                  ptr = (char *)mem; 
                  ltoa(buf2, (long)&ptr[i], 16);  
                }
                else if(strcmp(types1[choix], "SHORT") == 0)
                {
                  sptr = (short *)mem; 
                  ltoa(buf2, (long)&sptr[i], 16);
                }
#ifndef MCF5445X
                else if(strcmp(types1[choix], "DOUBLE") == 0)
                {
                  dptr = (double *)mem; 
                  ltoa(buf2, (long)&dptr[i], 16);
                }
#endif
                else
                  ltoa(buf2, (long)&mem[i], 16);
                a2p(conn, buf2);
                a2p(conn, "</b></td>");
              }
              if(i<max)
              { 
                if(ChoixMem && i == IndexMem)
                {
                  a2p(conn, "<td width=5%%><b>");
                  if(strcmp(types1[choix], "BYTE") == 0)
                  {
                    ptr = (char *)mem; 
                    ltoa(buf2, (long)ptr[i], 10);  
                  }
                  else if(strcmp(types1[choix], "SHORT") == 0)
                  {
                    sptr = (short *)mem; 
                    ltoa(buf2, (long)sptr[i], 10);
                  }
                  else if(strcmp(types1[choix], "LONG") == 0)
                    ltoa(buf2, mem[i], 10);
#ifndef MCF5445X
                  else if(strcmp(types1[choix], "FLOAT") == 0)
                    ftoa(*(float *)&mem[i], 2, buf2);
                  else if(strcmp(types1[choix], "DOUBLE") == 0)
                  {
                    dptr = (double *)mem; 
                    ftoa((float)dptr[i], 2, buf2);
                  }
#endif
                  else
                  {
                    buf2[0] = '$';
                    ltoa(&buf2[1], mem[i], 16);
                  }
                  a2p(conn, buf2);
                  a2p(conn, "</b></td>");
                }
                else
                {
                  a2p(conn, "<td width=5%%>");
                  if(strcmp(types1[choix], "BYTE") == 0)
                  {
                    ptr = (char *)mem; 
                    ltoa(buf2, (long)ptr[i], 10);  
                  }
                  else if(strcmp(types1[choix], "SHORT") == 0)
                  {
                    sptr = (short *)mem; 
                    ltoa(buf2, (long)sptr[i], 10);
                  }
                  else if(strcmp(types1[choix], "LONG") == 0)
                    ltoa(buf2, mem[i], 10);
#ifndef MCF5445X
                  else if(strcmp(types1[choix], "FLOAT") == 0)
                    ftoa(*(float *)&mem[i], 2, buf2);
                  else if(strcmp(types1[choix], "DOUBLE") == 0)
                  {
                    dptr = (double *)mem; 
                    ftoa((float)dptr[i], 2, buf2);
                  }
#endif
                  else
                  {
                    buf2[0] = '$';
                    ltoa(&buf2[1], mem[i], 16);                
                  }
                  a2p(conn, buf2);
                  a2p(conn, "</td>");
                }
              }
              else
                a2p(conn, "<td>-</td>");
              if((i&15)==15)
                a2p(conn, "\n</tr>\n");
            }
          }
          a2p(conn, "</table></form>\n");
          if(strcmp(types1[choix], "BIT") == 0)
            a2p(conn, mem_print2);
          else            
            a2p(conn, mem_print);
        }
      }
      choix++;
    }
  }
  else
  {
    a2p(conn, mem_html);
    a2p(conn, body);
  }
EndMem:
  a2p(conn, mem_end);
}

static void show_red_led(sock_conn *conn, char *request)
{
  (void)request;
  add_block(conn, (const char *)red_led_gif, sizeof(red_led_gif));
}

static void show_green_led(sock_conn *conn, char *request)
{
  (void)request;
  add_block(conn, (const char *)green_led_gif, sizeof(green_led_gif));
}

static void handle_screen(sock_conn *conn, char *req)
{
  (void)req;
  a2p(conn, screen_html);
  if(((info_fvdi != NULL) && (info_fvdi->var.xres_virtual > PICTURE_WIDTH) && (info_fvdi->var.yres_virtual > PICTURE_HEIGHT))
   || ((info_fvdi == NULL) && get_videl_base() && (get_videl_width() > PICTURE_WIDTH) && (get_videl_height() > PICTURE_WIDTH)))
  {
    char buf[10];
    a2p(conn, "X offset: <input type='text' value='");
    ltoa(buf, OffsetX, 10);
    a2p(conn, buf);
    a2p(conn, "' size=4 maxlength=4 name='OffsetX' id='OffsetX'>\n");
    a2p(conn, "&nbsp;&nbsp;Y offset: <input type='text' value='");
    ltoa(buf, OffsetY, 10);
    a2p(conn, buf);
    a2p(conn, "' size=4 maxlength=4 name='OffsetY' id='OffsetY'>\n");
    a2p(conn, "&nbsp;&nbsp;(screen ");
    if(info_fvdi != NULL)
    {
      ltoa(buf, info_fvdi->var.xres_virtual, 10);
      a2p(conn, buf);
      a2p(conn, "x");
      ltoa(buf, info_fvdi->var.yres_virtual, 10);
    }
    else
    {
      ltoa(buf, get_videl_width(), 10);
      a2p(conn, buf);
      a2p(conn, "x");
      ltoa(buf, get_videl_height(), 10);
    }
    a2p(conn, buf);
    a2p(conn, ")<br>\n");
  }
  else
  {
    a2p(conn, "<input type='hidden' value='0' name='OffsetX' id='OffsetX'>\n");
    a2p(conn, "<input type='hidden' value='0' name='OffsetY' id='OffsetY'>\n"); 
  }
  OffsetX = OffsetY = 0;
  a2p(conn, end_screen_html);
}  

static void handle_change_screen(sock_conn *conn, char *req)
{
  extern void call_mousevec(unsigned char *data, void (**mousevec)(void *));
  extern void call_ikbdvec(unsigned char code, void *iorec, void (**ikbdvec)());
  extern void (**mousevec)(void *);
  extern void *iorec;
  extern void (**ikbdvec)();
  extern unsigned long mousexy(void);
  extern unsigned char **keytbl;
#define STATE_RIGHT_SHIFT 0x01
#define STATE_LEFT_SHIFT  0x02
#define STATE_CTRL        0x04
#define STATE_ALT         0x08
#define STATE_CAPSLOCK    0x10
#define _sysbase          0x4F2 // header ROM
	unsigned char **header = (unsigned char **)*(long *)_sysbase;
  unsigned char sshift = *header[9];
  unsigned char *table_unshift = keytbl[0];
  unsigned char *table_shift = keytbl[1];
//  unsigned char *table_capslock = keytbl[2];
  char *ptr = strchr(req, '?');
  char *ptr2;
  if(ptr == NULL)
  	return;
  MouseX = MouseY = -1;
  MouseK = 0;
  if((ptr2 = (char *)strstr(ptr, "MouseX=")) != NULL)
    MouseX = (int)atol(&ptr2[7]);
  if((ptr2 = (char *)strstr(ptr, "MouseY=")) != NULL)
    MouseY = (int)atol(&ptr2[7]);
  if((ptr2 = (char *)strstr(ptr, "MouseK=")) != NULL)
    MouseK = (int)atol(&ptr2[7]);
  if((ptr2 = (char *)strstr(ptr, "OffsetX=")) != NULL)
    OffsetX = (int)atol(&ptr2[8]);
  if((ptr2 = (char *)strstr(ptr, "OffsetY=")) != NULL)
    OffsetY = (int)atol(&ptr2[8]);
  if(OffsetX < 0)
    OffsetX = 0;
  if(OffsetY < 0)
    OffsetY = 0;
  if(info_fvdi != NULL)
  {
    if(OffsetX > (info_fvdi->var.xres_virtual - PICTURE_WIDTH))
      OffsetX = info_fvdi->var.xres_virtual - PICTURE_WIDTH;
    if(OffsetY > (info_fvdi->var.yres_virtual - PICTURE_HEIGHT))
      OffsetY = info_fvdi->var.yres_virtual - PICTURE_HEIGHT;
  }
  else
  {
    if(OffsetX > (get_videl_width() - PICTURE_WIDTH))
      OffsetX = get_videl_width() - PICTURE_WIDTH;
    if(OffsetY > (get_videl_height() - PICTURE_WIDTH))
      OffsetY = get_videl_height() - PICTURE_HEIGHT;    
  }
  ptr2 = (char *)strstr(ptr, "Key=");
  if(((info_fvdi != NULL) && (info_fvdi->var.bits_per_pixel >= 16))
   || ((info_fvdi == NULL) && get_videl_base() && ((get_videl_bpp() == 4) || (get_videl_bpp() >= 16))))
  {
    if((ptr2 != NULL) && (iorec != NULL) && (ikbdvec != NULL))
    {
      ptr2 += 4;
      while((*ptr2 > ' ') && (*ptr2 != '&'))
  	  {
  	    char key = *ptr2++;
        unsigned char scancode = 0;
        int i, shift = 0;
        if(key == '+')
          key = ' ';
        if(key == '%')
        {
          i = (int)atol(ptr2);
          if(i < 127)
            key = (char)i;
          else
            key = 0;
          while((*ptr2 >= '0') && (*ptr2 <= '9'))
            ptr2++;
        }
        if(!key)
          continue; 
        for(i = 0; i < 128; i++)
        {
          if(table_unshift[i] == (unsigned char)key)
          {
            scancode = (unsigned char)i; 
            break;
          }
        }
        if(!scancode)
        {
          for(i = 0; i < 128; i++)
          {
            if(table_shift[i] == (unsigned char)key)
            {
              scancode = (unsigned char)i;
              shift = 1; 
              break;
            }
          }
        }
        if(scancode)
        {
          int level = asm_set_ipl(7);          /* mask interrupts for other tasks */
          if(sshift & STATE_CAPSLOCK)
            call_ikbdvec(0xBA, iorec, ikbdvec);
          if(shift && !(sshift & (STATE_LEFT_SHIFT|STATE_RIGHT_SHIFT)))
            call_ikbdvec(0x36, iorec, ikbdvec);
          else if(!shift)
          {
            if(sshift & STATE_RIGHT_SHIFT)
              call_ikbdvec(0xB6, iorec, ikbdvec);
            if(sshift & STATE_LEFT_SHIFT)
              call_ikbdvec(0xAA, iorec, ikbdvec);
          }
          call_ikbdvec(scancode, iorec, ikbdvec);
          call_ikbdvec(scancode + 0x80, iorec, ikbdvec);
          if(shift && !(sshift & (STATE_LEFT_SHIFT|STATE_RIGHT_SHIFT)))
            call_ikbdvec(0xB6, iorec, ikbdvec);
          else if(!shift)
          {
            if(sshift & STATE_RIGHT_SHIFT)
              call_ikbdvec(0x36, iorec, ikbdvec);
            if(sshift & STATE_LEFT_SHIFT)
              call_ikbdvec(0x2A, iorec, ikbdvec);
          }
          if(sshift & STATE_CAPSLOCK)
            call_ikbdvec(0x3A, iorec, ikbdvec);
          asm_set_ipl(level);        
        }
      }
    }
    if((MouseX >= 0) && (MouseY >= 0) && (MouseX < PICTURE_WIDTH) && (MouseY < PICTURE_HEIGHT) && (mousevec != NULL))
    {
      int level, i = 0, deltaX, deltaY;
      unsigned char new[4];      
      do
      {
        int curMouseX = (int)(mousexy() >> 16);
        int curMouseY = (int)(mousexy() & 0xffff);
        deltaX = MouseX + OffsetX - curMouseX;
        deltaY = MouseY + OffsetY - curMouseY;
        if(deltaX < -128)
          deltaX = -128;
        else if(deltaX > 127)
         deltaX = 127;
        if(deltaY < -128)
          deltaY = -128;
        else if(deltaY > 127)
          deltaY = 127;
        switch(MouseK)
        {
          case 1: new[0] = 0xFA; break;  /* left button */
          case 2: new[0] = 0xFB; break;  /* middle button */
          case 3: new[0] = 0xF9; break;  /* right button */
          default: new[0] = 0xF8; break; /* no button */
        }
        new[1] = (unsigned char)deltaX;
        new[2] = (unsigned char)deltaY;
        level = asm_set_ipl(7);          /* mask interrupts for other tasks */
        call_mousevec(new, mousevec);
        asm_set_ipl(level);
        i++;
      }
      while((i < 10) && (deltaX || deltaY));
    }
  }
}

static void show_screen(sock_conn *conn, char *request)
{
  unsigned char *input[PICTURE_HEIGHT];
  char *output = BufGif;
  Gif *gif;
  GifPalette *cmap;
  GifPicture *pic;
  GifBlock *block;
  static Colour col[256];
  int level, size, i, j, k, width, height, incr, bpp;
//	long mes = get_slice_timer();
  (void)request;
  if(info_fvdi != NULL)
  {
    width = incr = info_fvdi->var.xres_virtual;
    height = info_fvdi->var.yres_virtual;
    bpp = info_fvdi->var.bits_per_pixel;
  }
  else
  {
	  width = incr = (int)get_videl_width();
	  height = (int)get_videl_height();
	  bpp = (int)get_videl_bpp();
  }
  if(!(((info_fvdi != NULL) && (bpp >= 16))
   || ((info_fvdi == NULL) && get_videl_base() && ((bpp == 4) || (bpp >= 16)))))
  {
    add_block(conn, (const char *)red_led_gif, sizeof(red_led_gif));
    return;    
  }  
  if((BufGif != NULL) && (BufPic != NULL))
  {
    if(UseCurve)
    {
      if(SizeGif && (SecondBufGif != NULL))
      {
        level = asm_set_ipl(7);          /* mask interrupts for other tasks */
        add_block(conn, SecondBufGif, SizeGif);
        asm_set_ipl(level);  
      }
      else
        add_block(conn, (const char *)red_led_gif, sizeof(red_led_gif));
      return;  
    }  
    UseCurve = 1;
    if(width > PICTURE_WIDTH)
    	width = PICTURE_WIDTH;
    if(height > PICTURE_HEIGHT)
    	height = PICTURE_HEIGHT;
    switch(bpp)
    {
      case 4:
        {
          unsigned short *palette = (unsigned short *)get_videl_palette();
          long base = get_videl_base();
          if(palette != NULL)
          {
            for(i = 0; i < height; i++)
            {
              unsigned char *dst = (unsigned char *)&BufPic[width * i];
              unsigned short *src = (unsigned short *)(base + ((incr * (i + OffsetY)) >> 1));
              unsigned short *src2 = &src[OffsetX  & ~3];
              input[i] = dst;
              /* 4 interlaced planes to rgb332 conv */
              for(j = 0; j <  width; j += 16)
              {
                unsigned long plane0 = (unsigned long)*src2++;
                unsigned long plane1 = (unsigned long)*src2++;
                unsigned long plane2 = (unsigned long)*src2++;
                unsigned long plane3 = (unsigned long)*src2++;
                unsigned long mask = 0x8000;
                for(k = 15; k >= 0; k--)
                {
                  unsigned long index = 0, rgb, r, g, b;
                  if(plane0 & mask)
                    index |= 1;
                  if(plane1 & mask)
                    index |= 2;
                  if(plane2 & mask)
                    index |= 4;
                  if(plane3 & mask)
                    index |= 8;
                  mask >>= 1;
                  rgb = (unsigned long)palette[index] & 0x777;
                  r = (rgb >> 3) & 0xE0;
                  g = (rgb >> 2) & 0x1C;
                  b = (rgb >> 1) & 0x03;
                  *dst++ = (unsigned char)(r + g + b);
                }
              }
              src += (incr >> 2);
            }
          }
        }
        break;
      case 16:
        for(i = 0; i < height; i++)
        {
          unsigned short *src, *src2;
          unsigned char *dst = (unsigned char *)&BufPic[width * i];
          if(info_fvdi != NULL)
            src = (unsigned short *)&info_fvdi->screen_base[incr * sizeof(short) * (i + OffsetY)];
          else
            src = (unsigned short *)(get_videl_base() + (incr * sizeof(short) * (i + OffsetY)));
          src2 = &src[OffsetX];
          input[i] = dst;
          /* rgb565 to rgb332 conv */
          for(j = width - 1; j >= 0; j--)
          {
            unsigned long rgb = (unsigned long)*src2++;
            unsigned long r = (rgb >> 8) & 0xE0;
            unsigned long g = (rgb >> 6) & 0x1C;
            unsigned long b = (rgb >> 3) & 0x03;
            *dst++ = (unsigned char)(r + g + b);
          }
          src += incr;
        }
        break;
      case 32:
        for(i = 0; i < height; i++)
        {
          unsigned long *src, *src2;
          unsigned char *dst = (unsigned char *)&BufPic[width * i];
          if(info_fvdi != NULL)
            src = (unsigned long *)&info_fvdi->screen_base[incr * sizeof(long) * (i + OffsetY)];
          else
            src = (unsigned long *)(get_videl_base() + (incr * sizeof(long) * (i + OffsetY)));
          src2 = &src[OffsetX];
          input[i] = dst;
          /* rgb888 to rgb332 conv */
          for(j = width - 1; j >= 0; j--)
          {
            unsigned long rgb = *src2++;
            unsigned long r = (rgb >> 16) & 0xE0;
            unsigned long g = (rgb >> 11) & 0x1C;
            unsigned long b = (rgb >> 6) & 0x03;
            *dst++ = (unsigned char)(r + g + b);
          }
          src += incr;
        }
        break;
    }  
    /* Create a blank Gif: */
    gif = new_gif();
    if(gif != NULL)
    {
      /* Set the screen information: */
      gif->screen->width = width;
      gif->screen->height = height;
      gif->screen->has_cmap = 1;
      gif->screen->color_res = 8;
      gif->screen->cmap_depth = 8;
      /* Fill the colour map: */
      cmap = gif->screen->cmap;
      cmap->length = 256;
      cmap->colours = gif_alloc(cmap->length * sizeof(Colour));
      for(i=0; i < cmap->length; i++)
      {
        /* Append the colour: */
        col[i].alpha = 0;
        col[i].red = (char)((unsigned)i & 0xE0);
        col[i].green = (char)(((unsigned)i & 0x1C) << 3);
        col[i].blue = (char)(((unsigned)i & 3) << 6);
        cmap->colours[i] = col[i];
      }
      /* Create a new GifPicture: */
      pic = new_gif_picture();
      pic->width = width;
      pic->height = height;
      pic->interlace = 0; /* 0=no interlace, 1=interlace */
      pic->has_cmap = 0;
      pic->cmap_depth = 8; /* must be depth, despite not writing it */
      /* Link GifPicture data to supplied pixels: */
      pic->data = input;
      /* Link the GifPicture to the Gif: */
      block = new_gif_block();
      block->intro = 0x2C;
      block->pic = pic;
      /* Append the block: */
      size = ++gif->block_count;
      if(gif->blocks)
      {
        void *buf = gif->blocks;
        gif->blocks = (GifBlock **)pvPortMalloc2(size * sizeof(GifBlock *));
        memcpy(gif->blocks, buf, (size-1) * sizeof(GifBlock *));
        vPortFree2(buf);
      }
      else
        gif->blocks = (GifBlock **)pvPortMalloc2(size * sizeof(GifBlock *));
      gif->blocks[size-1] = block;
      /* Write the Gif file: */
      write_gif(&output, gif);
      /* Unlink the pixel data and clean up: */
      pic->data = NULL;
      del_gif(gif);
      size = (int)(output - BufGif);
      add_block(conn, BufGif, size);
//      board_printf("gif size %d bytes %ld us\r\n", size, (mes - get_slice_timer()) / 100UL);
      if(SecondBufGif != NULL)
      {
        level = asm_set_ipl(7);         /* mask interrupts for other tasks */
        memcpy(SecondBufGif, BufGif, size);
        SizeGif = size;
        asm_set_ipl(level);  
      }
      UseCurve = 0;
      return;
    }
    UseCurve = 1;
  }
}

#ifndef WEB_LIGHT

static void handle_curve(sock_conn *conn, char *req)
{
  float val;
  char buf[BUF_SIZE];
  unsigned long heure;
  int i,pos,flag;
  char *ptr, *ptr2;
  long *mem=NULL;
  OffsetPos = NB_COLONNES*(NB_IMAGES-1);
  ptr = strchr(req, '?');
  if(ptr == NULL)
    ptr = req;
  ptr2 = buf;
  i = 0;
  while(*ptr && (*ptr != ' ') && (i++ < BUF_SIZE-1))
    *ptr2++ = *ptr++;
  *ptr2 = 0;  
  if((ptr=(char *)strstr(buf, "ValeurMaxi=")) != NULL)
    ValeurMaxi = atof(&ptr[11]);
  if((ptr=(char *)strstr(buf, "ValeurMini=")) != NULL)
    ValeurMini = atof(&ptr[11]);
  if(ValeurMaxi <= ValeurMini)
  {
    val = ValeurMini;
    ValeurMini = ValeurMaxi;
    ValeurMaxi = val;
  }
  if((ptr=(char *)strstr(buf, "OffsetPos=")) != NULL)
  {
    OffsetPos = atol(&ptr[10]);
    OffsetPos /= (NB_COLONNES/2);
    OffsetPos *= (NB_COLONNES/2);
    if(OffsetPos<0 || OffsetPos>(NB_COLONNES*NB_IMAGES))
      OffsetPos = NB_COLONNES*(NB_IMAGES-1); 
  }
  if(strstr(buf, "STOP=STOP") != NULL)
  {
    TrigMem = CODE_STOP; /* arrt chantillonnage */
    HeureTrig = HeureMes;
  }
/*************************** COURBE *******************************************/
  heure = ((unsigned long)Tgettime() << 1);
  a2p(conn, curve_html);
  if((TrigMem != CODE_FIN_DECLENCH) && (TrigMem != CODE_STOP))
    a2p(conn, curve_refresh);
  else
  {
    a2p(conn, curve_script); // pour echelle
    ltoa(buf, NB_LIGNES/2, 10);
    a2p(conn, buf);
    a2p(conn, "; \n");
    a2p(conn, curve_end_script);
  }
  a2p(conn, curve_end_head);
  flag = 0;
  for(i=0;i<NB_COURBES;i++)
  {
    if((CMem[i] >= 0) && (IndexLect[i] != -1))
    {
      switch(i)
      {
        case 0: a2p(conn, "<font color=red>"); break; 
        case 1: a2p(conn, "<font color=magenta>"); break; 
        case 2: a2p(conn, "<font color=blue>"); break; 
        case 3: a2p(conn, "<font color=cyan>"); break;     
      }
      a2p(conn, types3[CMem[i]]);
      if(strcmp(types3[CMem[i]], "TASKS") != 0)
      {
        a2p(conn, " at $");
        ltoa(buf, IndexLect[i], 16);
        a2p(conn, buf);
      }
      a2p(conn, " </font>");
      flag = 1;
    }
  }
  if(!flag)
    a2p(conn, "Nothing");
  a2p(conn, "</h3></center></p>\n<p><table><tr>\n<td><font size=-1> ");
  ftoa(ValeurMaxi, ((ValeurMaxi > -100.0) && (ValeurMaxi < 100.0)) ? 1 : 0, buf);
  a2p(conn, buf);
  a2p(conn, " <br> ");
  ftoa(ValeurMini, ((ValeurMini > -100.0) && (ValeurMini < 100.0)) ? 1 : 0, buf);
  a2p(conn, buf);
  a2p(conn, " ("); 
  if((((ValeurMaxi-ValeurMini)/(float)NB_LIGNES) > -100.0)
   && (((ValeurMaxi-ValeurMini)/(float)NB_LIGNES) < 100.0))
   ftoa((ValeurMaxi-ValeurMini)/(float)NB_LIGNES, 1, buf);
  else
   ftoa((ValeurMaxi-ValeurMini)/(float)NB_LIGNES, 0, buf);
  a2p(conn, buf);
  a2p(conn, "/div) <i>");
  ltoa(buf, (long)OffsetPos, 10);
  a2p(conn, buf);
  a2p(conn, "</i>\n");
  pos = ((PICTURE_WIDTH*NB_IMAGES*PreTrig)/100) / (PICTURE_WIDTH/NB_COLONNES);
  pos -= (NB_COLONNES/2);
  if(pos < 0)
    pos = 0;
  pos /= (NB_COLONNES/2);
  pos *= (NB_COLONNES/2);
//  if((TrigMem != CODE_FIN_DECLENCH) && (TrigMem != CODE_STOP))
//    a2p(conn, curve_file);
//  else
    a2p(conn, curve_file_redraw);
  a2p(conn, "<i>");
  ltoa(buf, (long)OffsetPos + NB_COLONNES, 10);
  a2p(conn, buf);
  a2p(conn, "</i></font></td>\n<td width=18%%>\n");
  a2p(conn, curve_boutons);
  if((TrigMem == CODE_FIN_DECLENCH) || (TrigMem == CODE_STOP))
  { 
    a2p(conn, "Scale pts/div<br>");
    a2p(conn, "<input type='button' value='-' name='MoinsEch' onClick='document.forms[0].Echelle.selectedIndex--; calc_min_max(); submitform.submit();'>\n");
    a2p(conn, "<select name='Echelle' onChange='calc_min_max(); submitform.submit()'>\n");
    flag = 0;
    for(i = 0; i < (sizeof(table_echelle)/sizeof(float)); i++)
    { 
      val = table_echelle[i];
      if((((ValeurMaxi-ValeurMini)/(float)NB_LIGNES) <= val) && !flag)
      {
        flag = 1;
        a2p(conn, "<option selected>");
      }
      else
        a2p(conn, "<option>");      
      if((val > -0.1) && (val < 0.1))
        ftoa(val, 4, buf);
      else if((val > -1.0) && (val < 1.0))
        ftoa(val, 3, buf);
      else if((val > -10.0) && (val < 10.0))
        ftoa(val, 2, buf);
      else if((val > -100.0) && (val < 100.0))
        ftoa(val, 1, buf);
      else
        ftoa(val, 0, buf);
      a2p(conn, buf);
      a2p(conn, "</option>\n");
    }
    a2p(conn, "</select>\n<input type='button' value='+' name='PlusEch' onClick='document.forms[0].Echelle.selectedIndex++; calc_min_max(); submitform.submit();'>\n");
    a2p(conn, "<br><br>\n");
    a2p(conn, "Value max <br>");
    a2p(conn, "<input type='button' value='-' name='MoinsMaxi' onClick='document.forms[0].ValeurMaxi.selectedIndex--; submitform.submit();'>\n");
    a2p(conn, "<select name='ValeurMaxi' onChange='submitform.submit()'>\n");
    flag = 0;
    val = table_min_max[(sizeof(def_table_min_max)/sizeof(float))-1];
    for(i = 1; i < (sizeof(def_table_min_max)/sizeof(float))-1; i++)
    {
      if(table_min_max[i] == val)
        flag = 1; /* deux valeurs identiques (prslection et valeur menu) */ 
    }
    for(i = 1; i < (sizeof(def_table_min_max)/sizeof(float))-flag; i++)
    {
      val = table_min_max[i];
      if(val == ValeurMaxi)
        a2p(conn, "<option selected>");
      else
        a2p(conn, "<option>");      
      if((val > -100.0) && (val < 100.0))
        ftoa(val, 1, buf);
      else
        ftoa(val, 0, buf);
      a2p(conn, buf);
      if(i == ((sizeof(def_table_min_max)/sizeof(float))-1))
        a2p(conn, " M");
      a2p(conn, "</option>\n");
    }
    a2p(conn, "</select>\n<input type='button' value='+' name='PlusMaxi' onClick='document.forms[0].ValeurMaxi.selectedIndex++; submitform.submit();'>\n");
    a2p(conn, "<br><br>\nValue min <br>");
    a2p(conn, "<input type='button' value='-' name='MoinsMini' onClick='document.forms[0].ValeurMini.selectedIndex--; submitform.submit();'>\n");
    a2p(conn, "<select name='ValeurMini' onChange='submitform.submit()'>\n");
    flag = 0;
    val = table_min_max[0];
    for(i = 1; i < (sizeof(def_table_min_max)/sizeof(float))-1; i++)
    {
      if(table_min_max[i] == val)
        flag = 1; /* deux valeurs identiques (prslection et valeur menu) */ 
    }
    for(i = flag; i < (sizeof(def_table_min_max)/sizeof(float))-1;  i++)
    {
      val = table_min_max[i];
      if(val == ValeurMini)
        a2p(conn, "<option selected>");
      else
        a2p(conn, "<option>");      
      if((val > -100.0) && (val < 100.0))
        ftoa(val, 1, buf);
      else
        ftoa(val, 0, buf);
      a2p(conn, buf);
      if(i == 0)
        a2p(conn, " m");
      a2p(conn, "</option>\n");
    }
    a2p(conn, "</select>\n<input type='button' value='+' name='PlusMini' onClick='document.forms[0].ValeurMini.selectedIndex++; submitform.submit();'>\n");
    a2p(conn, "<br><br>\nOffset div <br>");
    a2p(conn, "<input type='button' value='-' name='Moins' onClick='document.forms[0].OffsetPos.selectedIndex--; submitform.submit();'>\n");
    a2p(conn, "<select name='OffsetPos' onChange='submitform.submit()'>\n");
    for(i=0; i<=(NB_COLONNES*(NB_IMAGES-1)); i+=(NB_COLONNES/2))
    {
      if(i == OffsetPos)
        a2p(conn, "<option selected>");
      else
        a2p(conn, "<option>");
      ltoa(buf, (long)i, 10);
      a2p(conn, buf);
      a2p(conn, "-");
      ltoa(buf, (long) i+NB_COLONNES, 10);
      a2p(conn, buf);
      if((i == pos) && (TrigMem != 0) && (TrigMem != CODE_STOP))
        a2p(conn, " T");
      a2p(conn, "</option>\n");
    }
    a2p(conn, "</select>\n<input type='button' value='+' name='Plus' onClick='document.forms[0].OffsetPos.selectedIndex++; submitform.submit();'>\n");
  }
  else
    a2p(conn, "<input type='submit' value='STOP' name='STOP'>\n");
  a2p(conn, "<br><br>\n");
  a2p(conn, curve_print);
  a2p(conn, "</td>\n</tr></table>\n<font size=-1><p align=right>");
  for(i=0; i<=(NB_COLONNES*(NB_IMAGES-1)); i+=(NB_COLONNES/2))
  {
    if(i == OffsetPos)
    {
      if((i == pos) && (TrigMem != 0) && (TrigMem != CODE_STOP)) /* dclenchement */
        a2p(conn, "[<font color=green>T</font>]");
      else
        a2p(conn, "[]");       
    }
    else
    {
      if((i == pos) && (TrigMem != 0) && (TrigMem != CODE_STOP)) /* dclenchement */
        a2p(conn, "<font color=green>T</font>");
      else
        a2p(conn, "-");
    }
  }
  if(TrigMem > 0)
  {
    a2p(conn, " - \n<font color=green>Trigger armed on ");
    a2p(conn, types2[TrigMem-1]);
    a2p(conn, " at $");
    ltoa(buf, IndexTrig, 16);
    a2p(conn, buf);
  }
  else if(TrigMem == CODE_DECLENCHE)
    a2p(conn, " - \n<font color=green>Trigger release</font> - ");
  else if(TrigMem == CODE_FIN_DECLENCH)
    a2p(conn, " - \n<font color=green>End trigger release</font> - ");
  else
    a2p(conn, " - ");    
  a2p(conn, "Value");
  if(!flag)
    a2p(conn, " Nothing ");
  for(i=0;i<NB_COURBES;i++)
  {
    if((CMem[i] >= 0) && (IndexLect[i] != -1))
    {
      switch(i)
      {
        case 0: a2p(conn, "<font color=red> "); break; 
        case 1: a2p(conn, "<font color=magenta> "); break; 
        case 2: a2p(conn, "<font color=blue> "); break; 
        case 3: a2p(conn, "<font color=cyan> "); break;     
      }
      mem = (long *)IndexLect[i];
      if(strcmp(types3[CMem[i]], "BYTE") == 0)
        ltoa(buf, (long)(*(char *)mem), 10);
      else if(strcmp(types3[CMem[i]], "SHORT") == 0)
        ltoa(buf, (long)(*(short *)mem), 10);
      else if(strcmp(types3[CMem[i]], "FLOAT") == 0)
        ftoa(*(float *)mem, 2, buf);
      else if(strcmp(types3[CMem[i]], "DOUBLE") == 0)
        ftoa((float)(*(double *)mem), 2, buf);
      else if(strcmp(types3[CMem[i]], "TASKS") == 0)
      {
        buf[0] = '?';
        buf[1] = '\0';
      }
      else
        ltoa(buf, *mem, 10);
      a2p(conn, buf);
      a2p(conn, "</font>");
    }
  }
  a2p(conn, " - Time ");
  ltoa(buf, (TrigMem == CODE_FIN_DECLENCH) ? HeureTrig >> 12 : heure >> 12, 10);
  a2p(conn, buf);
  a2p(conn, ":");
  ltoa(buf, (TrigMem == CODE_FIN_DECLENCH) ? (HeureTrig >> 6) & 0x3f : (heure >> 6) & 0x3f, 10);
  a2p(conn, buf);
  a2p(conn, ":");
  ltoa(buf, (TrigMem == CODE_FIN_DECLENCH) ? HeureTrig & 0x3f : heure & 0x3f, 10);
  a2p(conn, buf);
  a2p(conn, " - Time base in mS (");
  ltoa(buf, (PICTURE_WIDTH/NB_COLONNES) * (long)TempsC, 10);
  a2p(conn, buf);
  a2p(conn, "/div)</p></font>\n");
  a2p(conn, curve_end);
}

static void draw_line(int x1, int y1, int x2, int y2, int c)
{
  unsigned int pos = y1*PICTURE_WIDTH + x1;
  int count;
  int one_step, both_step;
  int d, incrE, incrNE;
  int dx, dy;
  int x_step = 1;
  int y_step = PICTURE_WIDTH;
  dx = x2 - x1;
  if(dx < 0)
  {
    dx = -dx;
    x_step = -x_step;
  }
  dy = y2 - y1;
  if(dy < 0)
  {
    dy = -dy;
    y_step = -y_step;
  }
  if(dx > dy)
  {
    count = dx;
    one_step = x_step;
    both_step = y_step;
    incrE = 2 * dy;
    incrNE = -2 * dx;
    d = -dx;
  }
  else
  {
    count = dy;
    one_step = y_step;
    both_step = x_step;
    incrE = 2 * dx;
    incrNE = -2 * dy;
    d = -dy;
  }
  if(BufPic != NULL)
  {
    for(; count >= 0; count--)
    {
      if(pos < PICTURE_WIDTH*PICTURE_HEIGHT)
        BufPic[pos] = c;
      d += incrE;
      if(d >= 0)
      {
        d += incrNE;
        pos += both_step;
      }
      pos += one_step;
    }
  }
}

static inline void plot(int x, int y, int c)
{
  if(BufPic != NULL)
    BufPic[PICTURE_WIDTH*y + x] = (char)c;
}

static void show_curve(sock_conn *conn, char *request)
{
  unsigned char *input[PICTURE_HEIGHT];
  char *output = BufGif;
  Gif *gif;
  GifPalette *cmap;
  GifPicture *pic;
  GifExtension *ext = NULL;
  unsigned char *data;
  GifBlock *block;
  static Colour col[256];
  short *pcolours = colours;
  int i, j, k, l, x, y, y0[NB_COURBES], c, size, step, posaff;
  float val[NB_COURBES];
  float fy;
  int level;
  (void)request;
  if((BufGif != NULL) && (BufPic != NULL))
  {
    if(UseCurve)
    {
      if(SizeGif && (SecondBufGif != NULL))
      {
        level = asm_set_ipl(7);          /* mask interrupts for other tasks */
        add_block(conn, SecondBufGif, SizeGif);
        asm_set_ipl(level);  
      }
      else
        add_block(conn, work_gif, sizeof(work_gif));
      return;  
    }  
    UseCurve = 1;
    for(i=0;i<PICTURE_HEIGHT;i++)
      input[i] = (unsigned char *)&BufPic[PICTURE_WIDTH*i];
    memset(BufPic, 0, PICTURE_WIDTH*PICTURE_HEIGHT);
    step = PICTURE_WIDTH/NB_COLONNES;
    /* tableau de couleurs RVB dans images.h */
    for(x=step;x<PICTURE_WIDTH;x+=step)  /* quadrillage vertical */
    {
      for(y=0;y<PICTURE_HEIGHT;y++)
      {
        if((y>>2)&1)
          plot(x, y, 1); /* noir */
        else
          plot(x, y, 0); /* transparent */
      }
    }
    step = PICTURE_HEIGHT/NB_LIGNES;
    for(y=step;y<PICTURE_HEIGHT;y+=step)  /* quadrillage horizontal */
    {
      for(x=0;x<PICTURE_WIDTH;x++)
      {
        if((x>>2)&1)
          plot(x, y, 1); /* noir */
        else
          plot(x, y, 0); /* transparent */
      }
    }
    /* cadre */
    for(y=0;y<PICTURE_HEIGHT;plot(0,y,1),plot(PICTURE_WIDTH-1,y,1),y++); /* noir */
    for(x=0;x<PICTURE_WIDTH;plot(x,0,1),plot(x,PICTURE_HEIGHT-1,1),x++); /* noir */
    /* ligne 0 */
    y = PICTURE_HEIGHT-1-(int)((-ValeurMini * (float)PICTURE_HEIGHT) / (ValeurMaxi-ValeurMini));
    posaff = OffsetPos*(PICTURE_WIDTH/NB_COLONNES);
    if((y>=0) && (y<PICTURE_HEIGHT))
      for(x=0;x<PICTURE_WIDTH;plot(x++, y, 8)); /* gris ligne 0 */
    /* trigger */
    if((TrigMem != 0) && (TrigMem != CODE_STOP))
    {
      x = ((PICTURE_WIDTH*NB_IMAGES*PreTrig)/100) - posaff;
      if(x>=0 && x<PICTURE_WIDTH)
        for(y=0;y<PICTURE_HEIGHT;plot(x, y++, 3)); /* vert, ligne verticale */
      fy = (float)(PICTURE_HEIGHT-1)-(((ValeurTrig-ValeurMini) * (float)PICTURE_HEIGHT) / (ValeurMaxi-ValeurMini));
      y = (int)fy;
      if((y>=0) && (y<PICTURE_HEIGHT))
      {
        y = (int)fy;
        for(x=0;x<PICTURE_WIDTH;x++)
        {
          if((x>>2)&1)
            plot(x, y, 3);  /* vert, ligne horizontale */
        }    
      }
    }
    /* curve */
    for(i=k=l=0;i<NB_COURBES;i++)
    {
      level = asm_set_ipl(7);        /* mask interrupts */
      posaff = OffsetPos*(PICTURE_WIDTH/NB_COLONNES) + PosBuf[i];
      if(posaff >= (PICTURE_WIDTH*NB_IMAGES))
        posaff -= (PICTURE_WIDTH*NB_IMAGES);
      if(posaff+PICTURE_WIDTH <= (PICTURE_WIDTH*NB_IMAGES))
        memcpy(&latch_Pts[l],&Pts[k+posaff],PICTURE_WIDTH*sizeof(float));
      else
      { /* buffer circulaire => copie en 2 parties */
        j = (PICTURE_WIDTH*NB_IMAGES) - posaff;
        if(j>0 && j<=PICTURE_WIDTH) /* dbut */
          memcpy(&latch_Pts[l],&Pts[k+posaff],j*sizeof(float));
        j = (posaff+PICTURE_WIDTH) - (PICTURE_WIDTH*NB_IMAGES);
        if(j>=0 && j<PICTURE_WIDTH) /* fin */
          memcpy(&latch_Pts[l+PICTURE_WIDTH-j],&Pts[k],j*sizeof(float));
      }
      asm_set_ipl(level);  
      k += PICTURE_WIDTH*NB_IMAGES;
      l += PICTURE_WIDTH;
    }
    for(i=j=0;i<NB_COURBES;i++)
    {
      if((CMem[i] >= 0) && (IndexLect[i] != -1))
      {
        val[i] = latch_Pts[j]-ValeurMini;
        fy = (float)(PICTURE_HEIGHT-1) - ((val[i] * (float)PICTURE_HEIGHT) / (ValeurMaxi-ValeurMini));
        if(fy < (float)-PICTURE_HEIGHT)
          fy = (float)-PICTURE_HEIGHT;
        else if(fy > (float)(PICTURE_HEIGHT*2))
          fy = (float)(PICTURE_HEIGHT*2);
        y0[i] = (int)fy;
      }
      j += PICTURE_WIDTH;
    }
    for(x=1;x<PICTURE_WIDTH;x++)
    {
      for(i=j=0;i<NB_COURBES;i++)    
      {
        if((CMem[i] >= 0) && (IndexLect[i] != -1))
        {
          val[i] = latch_Pts[j+x]-ValeurMini;
          fy = (float)(PICTURE_HEIGHT-1) - ((val[i] * (float)PICTURE_HEIGHT) / (ValeurMaxi-ValeurMini));
          if(fy < (float)-PICTURE_HEIGHT)
            fy = (float)-PICTURE_HEIGHT;
          else if(fy > (float)(PICTURE_HEIGHT*2))
            fy = (float)(PICTURE_HEIGHT*2);
          y = (int)fy;
          switch(i)
          {
            case 0: c = 2; break; /* rouge */
            case 1: c = 7; break; /* magenta */
            case 2: c = 4; break; /* bleu */
            case 3: c = 5; break; /* cyan */
            default: c = 1; break; /* noir */
          }
          draw_line(x-1, y0[i], x, y, c);
          y0[i] = y;
        }
        j += PICTURE_WIDTH;
      }
    }
    /* Create a blank Gif: */
    gif = new_gif();
    if(gif != NULL)
    {
      /* Set the screen information: */
      gif->screen->width = PICTURE_WIDTH;
      gif->screen->height = PICTURE_HEIGHT;
      gif->screen->has_cmap = 1;
      gif->screen->color_res = 8;
      gif->screen->cmap_depth = 8;
      /* Fill the colour map: */
      cmap = gif->screen->cmap;
      cmap->length = 256; /* 1 << depth */
      cmap->colours = gif_alloc(cmap->length * sizeof(Colour));
      for(i=0; i < cmap->length; i++)
      {
        /* Append the colour: */
        col[i].alpha = 0;
        col[i].red = (char)((((long)*pcolours++)<<8)/1001);
        col[i].green = (char)((((long)*pcolours++)<<8)/1001);
        col[i].blue = (char)((((long)*pcolours++)<<8)/1001);
        cmap->colours[i] = col[i];
        if(i != 0)
          continue;  /* not transparent */
        col[i].alpha = 128;
        /* Create transparent colour block: */
        if(ext)
          continue;  /* already made one */
        ext = new_gif_extension();
        ext->marker = 0xF9;
        ext->data = gif_alloc(sizeof(GifData *));
        ext->data[0] = new_gif_data(4);
        data = ext->data[0]->bytes;
        data[0] = 0x05;  /* was 0x01 (no disposal method) */
        data[1] = 0x00;
        data[2] = 0x00;
        data[3] = i;
        ext->data_count++;
        /* Link the transparency block to the Gif: */
        block = new_gif_block();
        block->intro = 0x21;
        block->ext = ext;
        /* Append the block: */
        size = ++gif->block_count;
        if(gif->blocks)
        {
          void *buf = gif->blocks;
          gif->blocks = (GifBlock **)pvPortMalloc2(size * sizeof(GifBlock *));
          memcpy(gif->blocks, buf, (size-1) * sizeof(GifBlock *));
          vPortFree2(buf);
        }
        else
          gif->blocks = (GifBlock **)pvPortMalloc2(size * sizeof(GifBlock *));
        gif->blocks[size-1] = block;
      }
      /* Create a new GifPicture: */
      pic = new_gif_picture();
      pic->width = PICTURE_WIDTH;
      pic->height = PICTURE_HEIGHT;
      pic->interlace = 0; /* 0=no interlace, 1=interlace */
      pic->has_cmap = 0;
      pic->cmap_depth = 8; /* must be depth, despite not writing it */
      /* Link GifPicture data to supplied pixels: */
      pic->data = input;
      /* Link the GifPicture to the Gif: */
      block = new_gif_block();
      block->intro = 0x2C;
      block->pic = pic;
      /* Append the block: */
      size = ++gif->block_count;
      if(gif->blocks)
      {
        void *buf = gif->blocks;
        gif->blocks = (GifBlock **)pvPortMalloc2(size * sizeof(GifBlock *));
        memcpy(gif->blocks, buf, (size-1) * sizeof(GifBlock *));
        vPortFree2(buf);
      }
      else
        gif->blocks = (GifBlock **)pvPortMalloc2(size * sizeof(GifBlock *));
      gif->blocks[size-1] = block;
      /* Write the Gif file: */
      write_gif(&output, gif);
      /* Unlink the pixel data and clean up: */
      pic->data = NULL;
      del_gif(gif);
      size = (int)(output - BufGif);
      add_block(conn, BufGif, size);
      if(SecondBufGif != NULL)
      {
        level = asm_set_ipl(7);         /* mask interrupts for other tasks */
        memcpy(SecondBufGif, BufGif, size);
        SizeGif = size;
        asm_set_ipl(level);  
      }
      UseCurve = 0;
      return;
    }
    UseCurve = 1;
  }
  add_block(conn, work_gif, sizeof(work_gif));
}

#endif

static void select_file(sock_conn *conn, char *request)
{
  (void)request;
  a2p(conn, select_html);
  a2p(conn, select_form);
  a2p(conn, select_end);
}

static void handle_file(sock_conn *conn, char *req)
{
  char *buffer, *ptr, *ptr2, *ptr3;
  char buf[BUF_SIZE], buf2[BUF_SIZE];
  unsigned long len = (unsigned long)conn->content_length;
  int err,i;
  a2p(conn, file_html);
  if((buffer = (char *)strstr(req, "\r\n\r\n")) == NULL) /* jump header */
    buffer = req;
  memcpy(buf, buffer, BUF_SIZE-1);
  buf[BUF_SIZE-1] = 0;
  minus(buf);
  if((ptr = (char *)strstr(buf, "content-disposition: ")) != NULL)
  {
    ptr2 = (char *)strstr(ptr-buf+buffer, "CheminFichier");
    if((ptr = (char *)strstr(ptr, "content-type: ")) != NULL)
    {
      ptr = ptr-buf+buffer;
      i=0;                            /* mime decoding light !!! */
      while((*ptr++ != '\n') && (i < 4096))
        i++;
      while((*ptr++ != '\n') && (i < 4096))
        i++;
      len -= (unsigned long)(ptr-buffer);
      ptr3 = ptr;                     /* beginning of file */
      if((i = strlen(conn->boundary)) != 0)
      {
        while(ptr3 < &ptr[len])
        {
          if(*ptr3 == '-')
          {
            if(memcmp(ptr3, conn->boundary, i) == 0)
            {
              len = (unsigned long)(ptr3-ptr); /* fix size */
              if(ptr3[-4]== '\r' &&  ptr3[-3]== '\n' && ptr3[-2]== '-' && ptr3[-1]== '-')
                len -= 4;
              break;
            }
          }
          ptr3++;             
        }
      }
      else /* normally unused in boundary found before */
      {
        while(ptr3 < &ptr[len])
        {
          if(*ptr3 == '-')
          {
            i=0;
            while(ptr3[i++] == '-');    /* boundary -------------XXXXXXXXX */
            if(i >= 20)
            {
              if(ptr3[-1] == '\n')
                ptr3--;
              if(ptr3[-1] == '\r')
                ptr3--;
              len = (unsigned long)(ptr3-ptr); /* fix size */
              break;
            }
          }
          ptr3++; 
        }
      }
      i=0;
      while(i<40) /* beginning of file */
      {
        if(((unsigned char)ptr[i])>=' '&& ((unsigned char)ptr[i])<='z'
         && ptr[i]!='<' && ptr[i]!='>' && ptr[i]!='/' && ptr[i]!=';')
          buf2[i] = ptr[i];
        else
          buf2[i] = ' ';
        i++;
      }
      buf2[i] = 0;
      buf[0] = 0;
      if(ptr2 != NULL) /* extract host path from POST string */
      {
        i=0;
        while((*ptr2++ != '\n') && (i < 4096))
          i++;
        while((*ptr2++ != '\n') && (i < 4096))
          i++;
        i=0;
        while((ptr2[i] != '\n') && (ptr2[i] != '\r') && (i < BUF_SIZE))
        {
          buf[i] = ptr2[i];
          i++;
        }
        buf[i] = 0;
      }
      /* chargement .HEX */
      Mem_Data = 0;
      Flash = Error = 0;
      sbuf_ptr = sbuf;
      lowest_address = 0xFFFFFFFF;
      hight_address = 0;
      i = strlen(buf);
      if((buf[i-3]=='h' && buf[i-2]=='e' && buf[i-1]=='x')
       || (buf[i-3]=='H' && buf[i-2]=='E' && buf[i-1]=='X')
       || (buf[i-3]=='s' && buf[i-2]=='1' && buf[i-1]=='9')
       || (buf[i-3]=='S' && buf[i-2]=='1' && buf[i-1]=='9'))
      {
        Mem_Data = (unsigned long)pvPortMalloc2(SIZE_TOS_FLASH);
        if(Mem_Data)
        {
          memset((void *)Mem_Data, -1, len);
          conv_rec(ptr, (long)len);
        }
        else
          Error = 1;
      }
      if(Error)
      {
        a2p(conn, "<body bgcolor=red>\n<h3>Srecord file .HEX error : ");
        a2p(conn, buf);
        err = (int)copy_file(buf, ptr, (long)len);
        a2p(conn, "</h3>\nBeginning of the file <b>[</b> ");
        a2p(conn, buf2);
        a2p(conn, " <b>]</b><br><br>\n");
        a2p(conn, buf);
        if(err==0)
          a2p(conn, "<br>File copied inside the disk<br>\n");
        a2p(conn, file_back);
      }
      else
      {
        if(Flash)
        {
          if((err=Programmation(lowest_address,hight_address,(lowest_address-FLASH_TOS_FIRE_ENGINE)+Mem_Data)) < 0)
          {
            a2p(conn, "<body bgcolor=red>\n<h3>Programming/verify error</h3>\n");
            a2p(conn, file_back);
          }
          else
          {
            a2p(conn, "<body bgcolor=white>\n<br>\n");
            a2p(conn, "<p><h3>System copied in flash : ");
            a2p(conn, buf);              
            a2p(conn, "</h3><br>\n");   
            a2p(conn, file_back);
          }
        }
        else
        {
          a2p(conn, "<body bgcolor=red>\n<h3>File not for the flash : ");
          a2p(conn, buf);         
          err = (int)copy_file(buf, ptr, (long)len);
          a2p(conn, "</h3><br>\n");
          if(err==0)
            a2p(conn, "<br>File copied inside the ram disk<br>\n");
          a2p(conn, file_back);
        }
      }
      if(Mem_Data)
        vPortFree2((void *)Mem_Data);
      a2p(conn, file_end);
      return;
    }
  }
  a2p(conn, "<body bgcolor=red>\n<h3>Content not valid</h3><br>\n");
  a2p(conn, file_back);
  a2p(conn, file_end);
}

#ifndef WEB_LIGHT
static void show_work(sock_conn *conn, char *request)
{
  (void)request;
  add_block(conn, work_gif, sizeof(work_gif));
}
#endif

static http_list m_http_list[] = {
  { "GET", "/index.html", handle_menu, "text/html"},
  { "GET", "/menu.html", handle_menu, "text/html"},
  { "GET", "/mem.html", handle_mem, "text/html" },
  { "GET", "/gif/red_led.gif", show_red_led, "image/gif" },
  { "GET", "/gif/green_led.gif", show_green_led, "image/gif" },
  { "GET", "/screen.html", handle_screen, "text/html" },
  { "POST", "/change_screen.html", handle_change_screen, "application/x-www-form-urlencoded" },
  { "GET", "/gif/screen.gif", show_screen, "image/gif" },
#ifndef WEB_LIGHT
  { "GET", "/curve.html", handle_curve, "text/html" },
  { "GET", "/gif/curve.gif", show_curve, "image/gif" },
  { "GET", "/gif/work.gif", show_work, "image/gif" },
#endif
  { "GET", "/sel_file.html", select_file, "text/html" },
  { "POST","/file.html", handle_file, "text/html" },
  { "GET", "/ ", handle_menu, "text/html" },
  { NULL, NULL, NULL, NULL }
};

static void handle_url(sock_conn *conn, char *request)
{
  int i = 0, size;
  char *ptr;
  char path[256];
  if((memcmp(request, "GET", 3) != 0) && (memcmp(request, "POST", 4) != 0))
  {
    send_status_code(conn, 501); /* not implemented */
    return;
  }
  while(m_http_list[i].url != NULL)
  {
    size = strlen(m_http_list[i].method);
    if(memcmp(request, m_http_list[i].method, size) == 0)
    {
      ptr = &request[size];
      if(*ptr == ' ')
        ptr++;
      if(memcmp(ptr, m_http_list[i].url, strlen(m_http_list[i].url)) == 0)
      {
        unsigned long time = gettime();
        send_status_code(conn, 200); /* OK */
        m_http_list[i].handle_url(conn, request);
        send_date(conn, "Date: ", gettime());
        write_string(conn, "Server: LWIP/1.2.0\r\nAllow: GET, POST\r\nConnection: close\r\n");
        send_content_type(conn, m_http_list[i].content_type);
        send_content_length(conn, conn->response_size);
        send_date(conn, "Last-Modified: ", time);
        write_string(conn, "\r\n");  
        /* write out the dynamically generated page */    
        write_socket(conn, conn->response, conn->response_size, 1); /* flush */
        return;
      }
    }
    i++;
  }
  if(memcmp(request, "GET", 3) == 0)
  {
    ptr = &request[3];
    if(*ptr == ' ')
      ptr++;
    if(strlen(ptr) < 200)
    {
      long handle, len;
      char *buf;
      strcpy(path, "C:\\html");
      strcat(path, ptr);
      i = 3;
      while(path[i])
      {
        if(path[i] == '/')
          path[i] = '\\';
        else if((path[i] == '?') || (path[i] == ' '))
        {
          path[i] = '\0';
          break;
        }
        else if((path[i] >= 'A') && (path[i] <= 'Z'))
          path[i] += 0x20; /* minus */
        i++;  
      }
      handle = Fopen(path, 0);
      if(handle >= 0)
      {
        len = Fseek(0, handle, 2);
        if(len > 0)
        {
          Fseek(0, handle, 0);
          buf = (char *)pvPortMalloc2(len);
          if(buf)
          {
            if(Fread(handle, len, buf) >= 0)
            {
              char *p = &path[strlen(path)-3];
              unsigned long time = gettime();
              send_status_code(conn, 200); /* OK */
              send_date(conn, "Date: ", gettime());
              write_string(conn, "Server: LWIP/1.2.0\r\nAllow: GET, POST\r\nConnection: close\r\n");
              if(strcmp(p, "htm") == 0)
                send_content_type(conn, "text/html");
              else if(strcmp(p, "css") == 0)
                send_content_type(conn, "text/css");
              else if(strcmp(p, "gif") == 0)
                send_content_type(conn, "image/gif");
              else if(strcmp(p, "jpg") == 0)
                send_content_type(conn, "image/jpeg");
              else if(strcmp(p, "png") == 0)
                send_content_type(conn, "image/png");
              else if(strcmp(p, "wav") == 0)
                send_content_type(conn, "audio/x-wav");
              else if(strcmp(p, "mpg") == 0)
                send_content_type(conn, "video/mpg");
              else if(strcmp(p, "pdf") == 0)
                send_content_type(conn, "application/pdf");
              else if(strcmp(p, "jar") == 0)
                send_content_type(conn, "application/x-java-archive");
              else
                send_content_type(conn, "application/octet-stream");
              send_content_length(conn, len);
              send_date(conn, "Last-Modified: ", time);
              write_string(conn, "\r\n");
              write_socket(conn, buf, len, 1);  /* flush */
              vPortFree2(buf);
              Fclose(handle);
              return;
            }
          }        
          vPortFree2(buf);
        }
        Fclose(handle);        
      }
    }
  }
  send_status_code(conn, 404); /* not found */
}

#ifndef WEB_LIGHT

static void HTTP_mesure(void *pvParameters)
{
  int i, j, count_trace=0;
  unsigned long *buffer_trace=NULL;
  float valeur;
  long *mem=NULL;
  unsigned long time=0;
  if(pvParameters);
  while(1)
  {
    time = ((unsigned long)Tgettime() << 1);
    HeureMes = time;
    if((TrigMem > 0) && TempsC)
    {
      mem = (long*)IndexTrig;
      if(strcmp(types3[TrigMem-1], "BYTE") == 0) 
        valeur = (float)(*(char *)mem);
      else if(strcmp(types3[TrigMem-1], "SHORT") == 0) 
        valeur = (float)(*(short *)mem);
      else if(strcmp(types3[TrigMem-1], "FLOAT") == 0)
      {
        if(isnan(*(float *)mem))
          valeur = 0.0;
        else
          valeur = *(float *)mem;      
      }
      else if(strcmp(types3[TrigMem-1], "DOUBLE") == 0)
      {
        if(isnan(*(double *)mem))
          valeur = 0.0;
        else
          valeur = (float)(*(double *)mem);
      }
      else
        valeur = (float)*mem;
      if(SignTrig)
      {
        if(valeur >= ValeurTrig)
        {
          OffsetTrig = 0;
          TrigMem = CODE_DECLENCHE;
          HeureTrig = HeureMes;       
        }
      }
      else
      {
        if(valeur <= ValeurTrig)
        {
          OffsetTrig = 0;
          TrigMem = CODE_DECLENCHE;
          HeureTrig = HeureMes;       
        }
      }
    }
    if((TrigMem != CODE_FIN_DECLENCH) && (TrigMem != CODE_STOP) && TempsC)
    {
      if(TrigMem == -1)
      {
        if(OffsetTrig < (PICTURE_WIDTH*NB_IMAGES*(100-PreTrig))/100)
          OffsetTrig++;
        else
          TrigMem = CODE_FIN_DECLENCH;
      }
      for(i=j=0;j<NB_COURBES;j++)
      {
        if((CMem[j] >= 0) && (IndexLect[j] != -1))
        {
          mem = (long *)IndexLect[j];
          if(strcmp(types3[CMem[j]], "TASKS") == 0)
          {
            if(buffer_trace == NULL)
            {
#define DEFAULT_TRACE_SIZE 0x10000
              if((buffer_trace = (unsigned long *)pvPortMalloc(DEFAULT_TRACE_SIZE)) != NULL)
              {
                vTaskStartTrace((char *)buffer_trace, DEFAULT_TRACE_SIZE);
                count_trace = ((PICTURE_WIDTH*NB_IMAGES)/100) / (TempsC ? TempsC : 1);
              }
            }
          }
          else if(strcmp(types3[CMem[j]], "BYTE") == 0)
            Pts[i + PosBuf[j]++] = (float)(*(char *)mem);
          else if(strcmp(types3[CMem[j]], "SHORT") == 0)
            Pts[i + PosBuf[j]++] = (float)(*(short *)mem);
          else if(strcmp(types3[CMem[j]], "FLOAT") == 0)
          {
            if(isnan(*(float *)mem))
              Pts[i + PosBuf[j]++] = 0.0;
            else
              Pts[i + PosBuf[j]++] = *(float *)mem;
          }
          else if(strcmp(types3[CMem[j]], "DOUBLE") == 0)
          {
            if(isnan(*(double *)mem))
              Pts[i + PosBuf[j]++] = 0.0;
            else
              Pts[i + PosBuf[j]++] = (float)(*(double *)mem);
          }
          else
            Pts[i + PosBuf[j]++] = (float)*mem;
          if(PosBuf[j] >= (PICTURE_WIDTH*NB_IMAGES))
            PosBuf[j]=0; /* buffer circulaire */
        }
        i += PICTURE_WIDTH*NB_IMAGES;
      }
      if(buffer_trace != NULL)
      {
        count_trace--;
        if(count_trace < 0)
        {
          int flag = -1;
          for(j=0;j<NB_COURBES;j++)
          {
            if((CMem[j] >= 0) && (IndexLect[j] != -1)
             && (strcmp(types3[CMem[j]], "TASKS") == 0))
            {
              if(flag >= 0)
                memcpy(&Pts[j*PICTURE_WIDTH*NB_IMAGES],&Pts[flag*PICTURE_WIDTH*NB_IMAGES],PICTURE_WIDTH*NB_IMAGES*sizeof(float));
              else
              {
                unsigned long tick, task, timer, ptick, ptask, ptimer;
                long len, time;
                float *p = &Pts[j*PICTURE_WIDTH*NB_IMAGES];
                int x = (PICTURE_WIDTH*NB_IMAGES) - 1;
                flag = j;
                len = (long)ulTaskEndTrace() / sizeof(long);
                ptimer = buffer_trace[--len];
                timer = ptimer + 10;
                task = buffer_trace[--len];
                tick = buffer_trace[--len];
                while(len > 0)
                {
                  time = (long)timer - (long)ptimer;  /* delta uS */
                  if((time % 10) < 5)
                    time /= 10;
                  else
                    time = (time / 10) + 1;
                  timer = ptimer;
                  ptimer = buffer_trace[--len];
                  ptask = buffer_trace[--len];
                  ptick = buffer_trace[--len];
                  valeur = (float)task;
                  while(--time >= 0)
                  {
                    x--;
                    if(x < 0)
                      break;
                    p[x] = valeur;
                  }
                  if((x >= 0) && (tick != ptick)) /* fix period */
                  {
                    unsigned long ptimer2 = ptimer;
                    int len2 = len;
                    p[x] = 65536.0;  /* top tick */
                    time = 0;        /* total delta / 10 */
                    timer = ptimer2;
                    ptimer2 = buffer_trace[--len2];
                    --len2;
                    if(buffer_trace[--len2] == ptick)
                    {
                      long total = 0;
                      while(len2 > 0)
                      {
                        time = (long)timer - (long)ptimer2;
                        if((time % 10) < 5)
                          time /= 10;
                        else
                          time = (time / 10) + 1;
                        total += time;
                        timer = ptimer2;
                        ptimer2 = buffer_trace[--len2];
                        --len2;
                        if(buffer_trace[--len2] != ptick)
                          break;
                      }
                      if(len2 > 0)
                        ptimer = (unsigned long)(total * 10); /* fix period */
                    }
                    timer = 1000000/configTICK_RATE_HZ; /* start period */
                  }
                  task = ptask;
                  tick = ptick;
                }
                if(x >= 0)
                  memset(p, 0, sizeof(float) * (x+1));
                vPortFree(buffer_trace);
                buffer_trace = NULL;
              }
            }
          }
        }
      }
    }
    vTaskDelay((TempsC*configTICK_RATE_HZ)/1000);
  }
}

#endif

/*
 * Process an incoming connection on port 80.
 *
 * This simply checks to see if the incoming data contains a GET request, and
 * if so sends back a single dynamically created page.  The connection is then
 * closed.
 */
static void vProcessConnection(void *pvParameters)
{
  sock_msg *msg;
  sock_conn conn;
  char *pcRxString, *pcRxStringMinus;
  char *ptr, *ptr2;
  int sock, len, size_header, i;
  int task = (int)pvParameters;
  if(task);
  conn.buf_write = (char *)pvPortMalloc2(WRITEBUF_SIZE);
  pcRxString = (char *)pvPortMalloc2(READBUF_SIZE+1);
  pcRxStringMinus = (char *)pvPortMalloc2(READBUF_SIZE+1);
  if((mbox == NULL) || (conn.buf_write == NULL)
   || (pcRxString == NULL) || (pcRxStringMinus == NULL))
  {
    while(1)
      vTaskDelay(1);
  }
  while(1)
  {
    if(!xQueueReceive(mbox, &msg, portMAX_DELAY))
      continue;
    sock = msg->sock;
    conn.response_max_size = WRITEBUF_SIZE;
    conn.response = (char *)pvPortMalloc2(conn.response_max_size);
    conn.content_length = size_header = 0;
    conn.errno = conn.response_size = conn.offset_buf_write = 0;
    conn.sock = sock;
    conn.boundary[0] = 0;
    if(conn.response != NULL)
    {
      /* We expect to immediately get data */
      if((len = read_socket(&conn, pcRxString, READBUF_SIZE, &size_header)) >= 0)
      {
        if(strstr(pcRxString, "HTTP/1.0") != NULL)
          conn.version = VERSION_10;
        else if(strstr(pcRxString, "HTTP/1.1") != NULL)
          conn.version = VERSION_11;
        else
          conn.version = VERSION_UNKNOWN;
        strcpy(pcRxStringMinus, pcRxString);
        minus(pcRxStringMinus);
        if((ptr = (char *)strstr(pcRxStringMinus, "content-length: ")) != NULL)
          conn.content_length  = (int)atol(&ptr[16]);
        if(conn.content_length)
        {
          if(((ptr = (char *)strstr(pcRxStringMinus, "content-type: multipart/form-data")) != NULL)
           && ((ptr = (char *)strstr(ptr, "boundary=")) != NULL))
          {
            ptr+=9;
            ptr2 = conn.boundary;
            i = 0;
            while(*ptr && (*ptr != '\r') && (*ptr != '\n') && (*ptr != ';') && (i < 255))
              *ptr2++ = *ptr++;
            *ptr2 = 0;
          }
          ptr = (char *)pvPortMalloc2(size_header+conn.content_length+1);
          if(ptr != NULL)
          {
            memcpy(ptr, pcRxString, len);
            if(conn.content_length-len+size_header > 0)
              read_socket(&conn, &ptr[len], conn.content_length-len+size_header, NULL);
          }
          ptr2 = ptr;
        }      
        else
        {
          ptr = NULL;
          ptr2 = pcRxString;
        }
        conn.errno = 0;
        handle_url(&conn, ptr2);
        if(ptr != NULL)
          vPortFree2(ptr);    
      }
      vPortFree2(conn.response);
    }
    vTaskDelay(10);
    close(sock);
  }
}

/*------------------------------------------------------------*/

void vBasicWEBServer(void *pvParameters)
{
  static sock_msg msg;
  sock_msg *pmsg;
  char name[5];
  int sock, newsock, len = sizeof(struct sockaddr_in);
  static struct sockaddr_in address;
  int i, task;
  if(pvParameters);
  if(i);
  SizeGif = 0;
  BufPic = gif_alloc(PICTURE_WIDTH*PICTURE_HEIGHT);
  BufGif = gif_alloc(PICTURE_WIDTH*PICTURE_HEIGHT);
  SecondBufGif = gif_alloc(PICTURE_WIDTH*PICTURE_HEIGHT);
  MouseX = MouseY = -1;
  MouseK = 0;
  OffsetX = OffsetY = 0;
#ifndef WEB_LIGHT
  UseCurve = 0;
  ValeurMini = -1000.0;
  ValeurMaxi = 1000.0;
  TempsC = 1000/configTICK_RATE_HZ;
  TrigMem = IndexTrig = 0;
  SignTrig = 1;
  ValeurTrig = 1.0;
  PreTrig = 50;
  OffsetPos = NB_COLONNES*(NB_IMAGES-1);
  OffsetTrig = 0;
  for(i = 0; i < NB_COURBES; CMem[i] = IndexLect[i] = -1, PosBuf[i] = 0, i++);
  table_min_max = pvPortMalloc2(sizeof(def_table_min_max));
  Pts = pvPortMalloc2(NB_COURBES*PICTURE_WIDTH*NB_IMAGES*sizeof(float));
  latch_Pts = pvPortMalloc2(NB_COURBES*PICTURE_WIDTH*sizeof(float));
  if((Pts != NULL) && (latch_Pts != NULL) && (table_min_max != NULL))
  {
    memset(Pts,0,NB_COURBES*PICTURE_WIDTH*NB_IMAGES*sizeof(float));
    memcpy(table_min_max, def_table_min_max, sizeof(def_table_min_max));
    xTaskCreate(HTTP_mesure, (void *)"HMES", configMINIMAL_STACK_SIZE, NULL, 10, NULL);
  }
  else
  {
    while(1)
      vTaskDelay(1);    
  }
#endif
  mbox = xQueueCreate(10, sizeof(sock_msg));
  for(task = 0; task < TASKS_SERVER; task++)
  {
    name[0] = 'H';
    name[1] = 'T';
    name[2] = '0' + (char)((task + 1) / 10);
    name[3] = '0' + (char)((task + 1) % 10);
    name[4] = '\0';
    xTaskCreate(vProcessConnection, (void *)name, configMINIMAL_STACK_SIZE, (void *)task, 9, NULL);
  }
  /* Create a new tcp connection handle */
  if((sock = socket(AF_INET, SOCK_STREAM, 0)) == -1)
  {
    while(1)
      vTaskDelay(1);
  }
  else
  {
    int error, optval;
    optval = 1;
    error = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (void*)&optval, sizeof(optval));
//    if(!error)
//    {
//      int sock_buf_size = SOCK_BUF_SIZE;
//      error = setsockopt(sock, SOL_SOCKET, SO_SNDBUF, (const void *)&sock_buf_size, sizeof(sock_buf_size));
//      if(!error)
//      {
//        sock_buf_size = SOCK_BUF_SIZE;
//        error = setsockopt(sock, SOL_SOCKET, SO_RCVBUF, (const void *)&sock_buf_size, sizeof(sock_buf_size));
        if(!error)
        {
          int alive = 1;
          error = setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (const void *)&alive, sizeof(alive));
        }
//      }
//    }
  }
  address.sin_family = AF_INET;
  address.sin_addr.s_addr = htonl(INADDR_ANY);
  address.sin_port = htons(webHTTP_PORT);
  if((bind(sock, (struct sockaddr *)&address, len) == -1)
   || (listen(sock, 100) == -1))
  {
    close(sock);
    while(1)
      vTaskDelay(1);
  }
  /* Loop forever */
  while(1)
  {
    len = sizeof(address);
    newsock = accept(sock, (struct sockaddr*)&address, &len);
    if(newsock != -1)
    {
      /* Service connection */
      if(mbox != NULL)
      {
        pmsg = &msg;
        pmsg->sock = newsock;
        xQueueSend(mbox, &pmsg, 100);
      }        
    }
    vTaskDelay(1);
  }  
}

#endif /* LWIP */
#endif /* NETWORK */
